package ru.yandex.webmaster3.viewer.http.searchquery;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.searchquery.QueryFilter;
import ru.yandex.webmaster3.core.searchquery.QueryGroup;
import ru.yandex.webmaster3.core.searchquery.QueryGroupId;
import ru.yandex.webmaster3.core.searchquery.QueryId;
import ru.yandex.webmaster3.core.searchquery.QueryUtils;
import ru.yandex.webmaster3.core.searchquery.viewer.QueryFilterConverter;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.searchquery.QueryGroupService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author aherman
 */
@WriteAction
@Category("searchquery")
@Component("/searchquery/group/save")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class SaveQueryGroupAction extends AbstractUserVerifiedHostAction<SaveQueryGroupRequest,
        SaveQueryGroupResponse> {
    private final QueryGroupService queryGroupService;
    private final AbtService abtService;

    @Override
    public SaveQueryGroupResponse process(SaveQueryGroupRequest request) {
        String groupIdStr = request.getGroupId();
        QueryGroup queryGroup;
        List<QueryFilter> filter = QueryFilterConverter.toFilter(request.getFilters());
        boolean isNew = false;

        //modify
        if (groupIdStr != null) {
            Optional<QueryGroupId> groupIdO = QueryGroupId.byGroupIdStr(request.getHostId(), groupIdStr);
            if (groupIdO.isEmpty()) {
                return new SaveQueryGroupResponse.NoSuchGroupResponse(groupIdStr);
            }
            QueryGroupId groupId = groupIdO.get();
            if (groupId.isSpecial()) {
                queryGroup = queryGroupService.fromSpecialGroup(groupId);
            } else {
                queryGroup = queryGroupService.getGroup(groupId);
                if (queryGroup == null) {
                    return new SaveQueryGroupResponse.NoSuchGroupResponse(groupId.getGroupId().toString());
                }
                if (!filter.isEmpty()) {
                    queryGroup = queryGroupService.changeGroupFilter(groupId, filter);
                }

                String groupName = StringUtils.trimToEmpty(request.getGroupName());
                if (!StringUtils.isEmpty(groupName)) {
                    var group = queryGroupService.getGroupByName(request.getHostId(), groupName);
                    boolean newName = !Objects.equals(queryGroupService.getGroup(groupId).getName(), groupName);
                    if (group != null && newName) {
                        return new SaveQueryGroupResponse.GroupAlreadyExistReponse(groupIdStr);
                    }
                    queryGroup = queryGroupService.changeGroupName(groupId, groupName);
                }
            }

        } else { //create new
            if (queryGroupService.isGroupLimitReached(request.getHostId())) {
                return new SaveQueryGroupResponse.GroupLimitReachedResponse();
            }

            String groupName = StringUtils.trimToEmpty(request.getGroupName());
            if (StringUtils.isEmpty(groupName)) {
                throw new WebmasterException("Group name is empty",
                        new WebmasterErrorResponse.IllegalParameterValueResponse(this.getClass(),
                                SaveQueryGroupRequest.PARAM_GROUP_NAME, null));
            }

            queryGroup = queryGroupService.getGroupByName(request.getHostId(), groupName);
            if (queryGroup != null) {
                return new SaveQueryGroupResponse.NormalResponse(queryGroup, false);
            }

            queryGroup = queryGroupService.addGroup(request.getHostId(), groupName, filter);
            isNew = true;
        }

        //modifications
        if (request.getRemoveQueries() != null) {
            queryGroupService.removeQueriesFromGroup(queryGroup.getQueryGroupId(),
                    QueryId.textToId(Arrays.asList(request.getRemoveQueries())), true);
        }

        if (request.getMoveGroupId() != null) {
            Optional<QueryGroupId> moveQueryGroupIdO = QueryGroupId.byGroupIdStr(request.getHostId(),
                    request.getMoveGroupId());
            if (moveQueryGroupIdO.isEmpty()) {
                return new SaveQueryGroupResponse.NoSuchGroupResponse(request.getMoveGroupId());
            }
            QueryGroupId moveToGroupId = moveQueryGroupIdO.get();

            Set<String> moveQueries = new HashSet<>(QueryUtils.CheckAndFixQueryStringUTF8(request.getMoveQueries()));
            if (!moveQueries.isEmpty()) {
                if (queryGroupService.isLimitReached(moveToGroupId, moveQueries.size())) {
                    int queriesLeft = queryGroupService.getQueriesLeft(moveToGroupId);
                    return new SaveQueryGroupResponse.QueryLimitReachedResponse(request.getMoveGroupId(), queriesLeft);
                }

                queryGroupService.addQueriesToGroup(moveToGroupId, moveQueries, false);
                queryGroupService.removeQueriesFromGroup(moveToGroupId, QueryId.textToId(moveQueries), false);
            }
        }

        Set<String> addQueries = new HashSet<>(QueryUtils.CheckAndFixQueryStringUTF8(request.getAddQueries()));
        if (!addQueries.isEmpty()) {
            if (queryGroupService.isLimitReached(queryGroup.getQueryGroupId(), addQueries.size())) {
                int queriesLeft = queryGroupService.getQueriesLeft(queryGroup.getQueryGroupId());
                return new SaveQueryGroupResponse.QueryLimitReachedResponse(queryGroup.getGroupId().toString(),
                        queriesLeft);
            }
            queryGroupService.addQueriesToGroup(queryGroup.getQueryGroupId(), addQueries, true);
        }

        return new SaveQueryGroupResponse.NormalResponse(queryGroup, isNew);
    }
}
