package ru.yandex.webmaster3.viewer.http.searchquery.recommended;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.recommendedquery.dao.HostsWithOpenedRecommendedYDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

/**
 * Created by ifilippov5 on 08.11.17.
 */
@Description("Открыты ли рекомендованные запросы для хоста")
@ReadAction
@Category("recommended")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class GetIsOpenedRecommendedForHostAction extends AbstractUserVerifiedHostAction<GetIsOpenedRecommendedForHostAction.Req, GetIsOpenedRecommendedForHostAction.Res> {

    private static final int MAX_DURATION_CALCULATING_RECOMMENDED_DAYS = 5;
    private final HostsWithOpenedRecommendedYDao hostsWithOpenedRecommendedYDao;

    @Override
    public Res process(Req request) {
        Pair<WebmasterHostId, DateTime> openedHostInfo;
        try {
            openedHostInfo = hostsWithOpenedRecommendedYDao.getOpenDate(request.getHostId());
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
        if (openedHostInfo == null) {
            return new Res.NormalResponse(false, false);
        }
        boolean opened = (openedHostInfo.getLeft() != null);
        DateTime openDate = openedHostInfo.getRight();
        boolean inProcess = false;
        if (openDate != null) {
            DateTime boundDate = DateTime.now().minusDays(MAX_DURATION_CALCULATING_RECOMMENDED_DAYS);
            inProcess = (openDate.isAfter(boundDate));
        }
        return new Res.NormalResponse(opened, inProcess);
    }

    public static class Req extends AbstractUserVerifiedHostRequest {}

    public static abstract class Res implements ActionResponse {
        public static class NormalResponse extends Res implements ActionResponse.NormalResponse {
            private final boolean opened;
            private final boolean inProgress;

            public NormalResponse(boolean opened, boolean inProgress) {
                this.opened = opened;
                this.inProgress = inProgress;
            }

            public boolean isOpened() {
                return opened;
            }

            public boolean isInProgress() {
                return inProgress;
            }
        }
    }
}
