package ru.yandex.webmaster3.viewer.http.searchquery.recommended;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.AbstractFilter;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.recommendedquery.RecommendedQueriesService;
import ru.yandex.webmaster3.storage.recommendedquery.dao.RecommendedQueriesCHDao.Include;
import ru.yandex.webmaster3.storage.recommendedquery.filtres.RecommendedQueryConditions;
import ru.yandex.webmaster3.storage.recommendedquery.filtres.RecommendedQueryField;
import ru.yandex.webmaster3.storage.recommendedquery.filtres.RecommendedQueryFilter;
import ru.yandex.webmaster3.storage.recommendedquery.samples.RecommendedQuery;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.BoolOpCondition;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.viewer.http.AbstractUrlFilteringAction;

/**
 * Created by ifilippov5 on 09.03.17.
 */
@Description("Получить информацию о рекомендованных запросах")
@ReadAction
@Category("recommended")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class GetRecommendedQueryInfoAction extends AbstractUrlFilteringAction<GetRecommendedQueryInfoRequest, GetRecommendedQueryInfoResponse> {

    //WMC-11628: [Бэк] Скрывать рекомендованные запросы для хостов из blacklist
    private static final String HIDE_EXP_NAME = "HIDE_RECOMMENDED_QUERIES";

    private final AbtService abtService;
    private final RecommendedQueriesService recommendedQueriesService;

    @Override
    public GetRecommendedQueryInfoResponse process(GetRecommendedQueryInfoRequest request) {
        recommendedQueriesService.updateLastVisitDate(request.getHostId());
        Condition condition = makeCondition(request.getParameterFilters(), request.getRegionId());
        boolean hide = abtService.isInExperiment(request.getHostId(), HIDE_EXP_NAME);
        if (hide || !recommendedQueriesService.hasQueries(request.getHostId(), Include.BOTH)) {
            return new GetRecommendedQueryInfoResponse(0, Collections.emptyList(), false);
        }
        Include include = request.getInclude();
        long queriesCount = recommendedQueriesService.getQueriesCount(request.getHostId(), condition, include);
        if (queriesCount == 0) {
            return new GetRecommendedQueryInfoResponse(queriesCount, Collections.emptyList(), true);
        }
        List<RecommendedQuery> queries = recommendedQueriesService.getQueries(
                request.getHostId(), condition, request.getOrderBy(), request.getOrderDirection(),
                request.getP() * request.getPSize(), request.getPSize(), include
        );
        queries = queries.stream().map(RecommendedQuery::toRoundedSample).collect(Collectors.toList());
        return new GetRecommendedQueryInfoResponse(queriesCount, queries, true);
    }

    public static Condition makeCondition(List<? extends AbstractFilter<RecommendedQueryField>> filters, Long regionId) {
        // TODO temp fix
        if (regionId != null) {
            if (filters == null) {
                filters = new ArrayList<>();
            }
            // добавляем фильтр с регионом
            RecommendedQueryFilter regionFilter = new RecommendedQueryFilter(RecommendedQueryField.REGION_ID,
                    AbstractFilter.Operation.EQUAL, regionId.toString(), BoolOpCondition.Operator.AND);
            // TODO hack
            ((List) filters).add(regionFilter);
        }
        if (filters != null && !filters.isEmpty()) {
            return Condition.and(filters.stream().map(RecommendedQueryConditions::fromFilter).collect(Collectors.toList()));
        }
        return Condition.trueCondition();
    }
}
