package ru.yandex.webmaster3.viewer.http.searchquery.rivals;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.collect.Range;
import org.joda.time.LocalDate;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.core.searchquery.viewer.ViewerQueryGroup;
import ru.yandex.webmaster3.viewer.http.searchquery.statistic.AbstractQueryStatisticsResponse;

import java.util.Collections;
import java.util.List;

/**
 * @author aherman
 */
public abstract class GetRivalsStatisticsResponse extends AbstractQueryStatisticsResponse implements ActionResponse {

    public static final class NormalResponse extends GetRivalsStatisticsResponse implements ActionResponse.NormalResponse {

        private final int rangesCount;
        private final List<RivalsStatistic> statistics;

        public NormalResponse(int rangesCount, List<RivalsStatistic> statistics) {
            this.rangesCount = rangesCount;
            this.statistics = statistics;
        }

        public int getRangesCount() {
            return rangesCount;
        }

        public List<RivalsStatistic> getStatistics() {
            return statistics;
        }
    }

    public static final class RivalsStatistic {

        private final ViewerQueryGroup group;
        private final DateRange range;
        private final List<RivalsIndicator> indicators;

        public RivalsStatistic(ViewerQueryGroup group, DateRange range, List<RivalsIndicator> indicators) {
            this.group = group;
            this.range = range;
            this.indicators = indicators;
        }

        @JsonIgnore
        public ViewerQueryGroup getGroup() {
            return group;
        }

        public DateRange getRange() {
            return range;
        }

        public List<RivalsIndicator> getIndicators() {
            return indicators;
        }
    }

    public static final class RivalsIndicator {

        private final QueryIndicator name;
        private final List<ValueRange> ranges; // костыли для фронта

        public RivalsIndicator(QueryIndicator name, Double value, Double difference) {
            this.name = name;
            this.ranges = Collections.singletonList(new ValueRange(value, difference));
        }

        public QueryIndicator getName() {
            return name;
        }

        public List<ValueRange> getRanges() {
            return ranges;
        }
    }

    public static final class ValueRange {

        private final Double value;
        private final Double difference;

        public ValueRange(Double value, Double difference) {
            this.value = value;
            this.difference = difference;
        }

        @JsonInclude(JsonInclude.Include.ALWAYS)
        public Double getValue() {
            return value;
        }

        @JsonInclude(JsonInclude.Include.ALWAYS)
        public Double getDifference() {
            return difference;
        }
    }

    public static final class DateRange {
        private final LocalDate dateFrom;
        private final LocalDate dateTo;
        private final boolean wholeUserRange;

        public DateRange(LocalDate dateFrom, LocalDate dateTo, boolean wholeUserRange) {
            this.dateFrom = dateFrom;
            this.dateTo = dateTo;
            this.wholeUserRange = wholeUserRange;
        }

        public DateRange(Range<LocalDate> range, boolean wholeUserRange) {
            this(range.lowerEndpoint(), range.upperEndpoint(), wholeUserRange);
        }

        public LocalDate getDateFrom() {
            return dateFrom;
        }

        public LocalDate getDateTo() {
            return dateTo;
        }

        public boolean isWholeUserRange() {
            return wholeUserRange;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            DateRange dateRange = (DateRange) o;

            if (!dateFrom.equals(dateRange.dateFrom)) return false;
            return dateTo.equals(dateRange.dateTo);
        }

        @Override
        public int hashCode() {
            int result = dateFrom.hashCode();
            result = 31 * result + dateTo.hashCode();
            return result;
        }
    }

}
