package ru.yandex.webmaster3.viewer.http.searchquery.statistic;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.jetbrains.annotations.NotNull;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.proto.ProtobufUtils;
import ru.yandex.webmaster3.core.searchquery.QueryGroupId;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.core.searchquery.viewer.QueryGroupConverter;
import ru.yandex.webmaster3.core.searchquery.viewer.ViewerQueryGroup;
import ru.yandex.webmaster3.proto.dashboard.cache.DashboardCache;
import ru.yandex.webmaster3.storage.cache.DashboardCacheService;
import ru.yandex.webmaster3.storage.cache.DashboardType;
import ru.yandex.webmaster3.storage.http.searchquery.statistic.StatisticsHelper;
import ru.yandex.webmaster3.storage.searchquery.AggregatePeriod;
import ru.yandex.webmaster3.storage.searchquery.GroupStatisticsService2;
import ru.yandex.webmaster3.storage.searchquery.QueryGroupService;
import ru.yandex.webmaster3.viewer.util.DashboardUtils;

/**
 * @author aherman
 */
@ReadAction
@Category("searchquery")
public class GetGroupHistoryDashboardAction extends GetGroupHistoryAction {
    private static final Logger log = LoggerFactory.getLogger(GetGroupHistoryDashboardAction.class);

    private static final DashboardType CACHE_TYPE = DashboardType.SEARCH_QUERY_GROUP_HISTORY;

    private final DashboardCacheService dashboardCacheService;

    @Autowired
    public GetGroupHistoryDashboardAction(QueryGroupService queryGroupService,
                                          GroupStatisticsService2 groupStatisticsService2,
                                          DashboardCacheService dashboardCacheService) {
        super(queryGroupService, groupStatisticsService2);
        this.dashboardCacheService = dashboardCacheService;
    }

    @Override
    public GroupsStatisticResponse process(GetGroupHistoryRequest request) {
        final LocalDate userRangeStart = request.getLocalDateFrom();
        final LocalDate userRangeEnd = request.getLocalDateTo();

        final List<QueryIndicator> indicators = StatisticsHelper.asList(7, request.getIndicator());
        final List<QueryGroupId> groupIds = StatisticsHelper.asGroupIds(7, request.getHostId(),
                request.getGroupId());
        final Set<Integer> regionIds = Collections.emptySet();
        final AggregatePeriod period = request.getPeriod();

        String cacheKey = String.valueOf(request.hashCode());
        try {
            Optional<List<GroupsStatisticResponse.GroupStat>> value = dashboardCacheService
                    .getData(request.getHostId(), CACHE_TYPE, cacheKey, getDeserializer(request));
            if (value.isPresent()) {
                log.debug("From cache: {} {} {}", request.getHostId(), CACHE_TYPE, cacheKey);
                return new GroupsStatisticResponse.NormalResponse(null, value.get());
            }
        } catch (Exception e) {
            log.error("Unable to read cached value: {} {} {}", request.getHostId(), CACHE_TYPE, cacheKey, e);
        }

        List<GroupsStatisticResponse.GroupStat> result = getStatistics(request.getHostId(), groupIds, indicators,
                userRangeStart, userRangeEnd, period, request.getRegionInclusion(), regionIds, request.getDeviceType()
        );
        try {
            log.debug("Update cache: {} {} {}", request.getHostId(), CACHE_TYPE, cacheKey);
            dashboardCacheService.saveData(request.getHostId(), DashboardType.SEARCH_QUERY_GROUP_HISTORY, cacheKey, serialize(result));
        } catch (Exception e) {
            log.error("Unable to update cached value: {} {} {}", request.getHostId(), CACHE_TYPE, cacheKey, e);
        }

        return new GroupsStatisticResponse.NormalResponse(null, result);
    }

    @NotNull
    private DashboardCacheService.Mapper<List<GroupsStatisticResponse.GroupStat>> getDeserializer(
            GetGroupHistoryRequest request) {
        return (valueBytes) -> {
            DashboardCache.GroupHistoryDashboardCache dashboardCache;
            try {
                dashboardCache = DashboardCache.GroupHistoryDashboardCache.parseFrom(valueBytes);
            } catch (IOException e) {
                log.error("Unable to read value", e);
                return Optional.empty();
            }

            List<GroupsStatisticResponse.GroupStat> result = new ArrayList<>();
            for (int i = 0; i < dashboardCache.getHistoriesCount(); i++) {
                DashboardCache.GroupHistoryDashboardCache.GroupHistory history =
                        dashboardCache.getHistories(i);
                List<QueryGroupId> queryGroupIds =
                        StatisticsHelper.asGroupIds(1, request.getHostId(), history.getGroupId());
                if (queryGroupIds.isEmpty()) {
                    continue;
                }
                QueryGroupId groupId = queryGroupIds.get(0);
                if (!groupId.isSpecial()) {
                    continue;
                }
                ViewerQueryGroup viewerQueryGroup =
                        QueryGroupConverter.toViewerGroup(groupId.getSpecialGroup());
                List<AbstractQueryStatisticsResponse.RangeStat> rangeStats = new ArrayList<>();
                for (int j = 0; j < history.getRangeStatCount(); j++) {
                    DashboardCache.RangeStatCache rs = history.getRangeStat(j);
                    rangeStats.add(new AbstractQueryStatisticsResponse.RangeStat(
                                    DashboardUtils.intToLocalDate(rs.getDateFrom()),
                                    DashboardUtils.intToLocalDate(rs.getDateTo()),
                                    ProtobufUtils.getDouble(rs.hasValue(), rs.getValue()),
                                    ProtobufUtils.getDouble(rs.hasDifference(), rs.getDifference())
                            )
                    );
                }
                result.add(new GroupsStatisticResponse.GroupStat(viewerQueryGroup,
                        new AbstractQueryStatisticsResponse.IndicatorStats(
                                QueryIndicator.R.fromValueOrUnknown(history.getQueryIndicator()),
                                rangeStats
                        )));
            }
            return Optional.of(result);
        };
    }

    @NotNull
    private byte[] serialize(List<GroupsStatisticResponse.GroupStat> result) {
        DashboardCache.GroupHistoryDashboardCache.Builder ghdc = DashboardCache.GroupHistoryDashboardCache.newBuilder();

        for (GroupsStatisticResponse.GroupStat groupStat : result) {
            DashboardCache.GroupHistoryDashboardCache.GroupHistory.Builder gh =
                    DashboardCache.GroupHistoryDashboardCache.GroupHistory.newBuilder();

            String groupId = groupStat.getGroup().getGroupId();
            AbstractQueryStatisticsResponse.IndicatorStats indicatorStats = groupStat.getIndicator();
            QueryIndicator indicator = indicatorStats.getName();
            gh.setGroupId(groupId);
            gh.setQueryIndicator(indicator.value());
            for (GroupsStatisticResponse.RangeStat rangeStat : indicatorStats.getRanges()) {
                DashboardCache.RangeStatCache.Builder rs = DashboardCache.RangeStatCache.newBuilder()
                        .setDateFrom(DashboardUtils.localDateToInt(rangeStat.getDateFrom()))
                        .setDateTo(DashboardUtils.localDateToInt(rangeStat.getDateTo()));
                if (rangeStat.getValue() != null) {
                    rs.setValue(rangeStat.getValue());
                }
                if (rangeStat.getDifference() != null) {
                    rs.setDifference(rangeStat.getDifference());
                }
                gh.addRangeStat(rs);
            }
            ghdc.addHistories(gh);
        }

        return ghdc.build().toByteArray();
    }

}
