package ru.yandex.webmaster3.viewer.http.searchquery.statistic;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.searchquery.QueryId;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.core.searchquery.viewer.ViewerQuery;
import ru.yandex.webmaster3.storage.http.searchquery.statistic.StatisticsHelper;
import ru.yandex.webmaster3.storage.searchquery.AccumulatorMap;
import ru.yandex.webmaster3.storage.searchquery.QueryStat;
import ru.yandex.webmaster3.storage.searchquery.QueryStatisticsService2;
import ru.yandex.webmaster3.storage.searchquery.RangeFactory;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author aherman
 */
@ReadAction
@Category("searchquery")
public class GetQueryPieChartAction extends AbstractUserVerifiedHostAction<GetQueryPieChartRequest, QueryStatisticsResponse> {
    private QueryStatisticsService2 queryStatisticsService2;

    @Override
    public QueryStatisticsResponse process(GetQueryPieChartRequest request) throws WebmasterException {
        final LocalDate userRangeStart = request.getLocalDateFrom();
        final LocalDate userRangeEnd = request.getLocalDateTo();
        final List<QueryId> queryIds = StatisticsHelper.asQueryIds(7, request.getQueryId());
        final Set<Integer> regions = StatisticsHelper.asSet(10, request.getRegionId());
        final QueryIndicator indicator = request.getIndicator();
        final List<QueryIndicator> indicators = Collections.singletonList(indicator);
        if (queryIds.isEmpty()) {
            throw new WebmasterException("Error",
                    new WebmasterErrorResponse.IllegalParameterValueResponse(this.getClass(), "queryId", null));
        }


        RangeSet<LocalDate> rangeSet = RangeFactory.singleRange(userRangeStart, userRangeEnd);
        Map<QueryId, String> queryTexts =
                queryStatisticsService2.getQueryTexts(request.getHostId(), request.getSpecialGroup(), queryIds);
        List<QueryStat> statistics = queryStatisticsService2
                .getStatistics(request.getSpecialGroup(),
                        request.getHostId(),
                        indicators, queryIds,
                        regions, request.getRegionInclusion(),
                        rangeSet.span().lowerEndpoint(), rangeSet.span().upperEndpoint(), request.getDeviceType());

        List<QueryStatisticsResponse.QueryStat> result = new ArrayList<>();

        Map<QueryId, AccumulatorMap> accumulators = new HashMap<>();
        Function<QueryId, AccumulatorMap> accumulatorFactory =
                q -> {
                    return AccumulatorMap.create(indicators, rangeSet);
                };
        for (QueryStat queryStat : statistics) {
            AccumulatorMap ac = accumulators.computeIfAbsent(queryStat.getQueryId(), accumulatorFactory);
            ac.apply(queryStat);
        }

        for (QueryId queryId : queryIds) {
            String queryText = queryTexts.get(queryId);
//            List<QueryStat> qStat =
//                    statistics.stream().filter(qs -> qs.getQueryId().equals(queryId)).collect(Collectors.toList());
//            AccumulatorMap accumulatorMap = AccumulatorMap.create(Collections.singletonList(indicator), rangeSet);
//            qStat.forEach(accumulatorMap::apply);

            AccumulatorMap accumulatorMap = accumulators.computeIfAbsent(queryId, accumulatorFactory);

            List<Pair<Range<LocalDate>, Double>> indicatorStat = accumulatorMap.getIndicator(indicator);
            Pair<Range<LocalDate>, Double> rangeStat = indicatorStat.get(0);

            result.add(new QueryStatisticsResponse.QueryStat(new ViewerQuery(queryId, queryText),
                    new AbstractQueryStatisticsResponse.IndicatorStats(indicator, Collections.singletonList(
                            new AbstractQueryStatisticsResponse.RangeStat(
                                    rangeStat.getKey().lowerEndpoint(),
                                    rangeStat.getKey().upperEndpoint(),
                                    rangeStat.getValue(),
                                    null)
                    ))));
        }

        return new QueryStatisticsResponse.NormalResponse(null, null, result);
    }

    @Required
    public void setQueryStatisticsService2(QueryStatisticsService2 queryStatisticsService2) {
        this.queryStatisticsService2 = queryStatisticsService2;
    }
}
