package ru.yandex.webmaster3.viewer.http.searchquery.statistic;

import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.searchquery.*;
import ru.yandex.webmaster3.core.searchquery.viewer.ViewerPath;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.storage.http.searchquery.statistic.StatisticsHelper;
import ru.yandex.webmaster3.storage.searchquery.*;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

import java.util.*;
import java.util.function.Function;

/**
 * @author lester
 */
@ReadAction
@Category("searchquery")
public class GetUrlListStatisticsAction extends
        AbstractUserVerifiedHostAction<GetUrlListStatisticsRequest, GetUrlListStatisticsResponse> {

    @Autowired
    private UrlStatisticsService2 urlStatisticsService2;

    @Override
    public GetUrlListStatisticsResponse process(GetUrlListStatisticsRequest request) throws WebmasterException {
        final SpecialGroup specialGroup = request.getSpecialGroup();
        final LocalDate userRangeStart = request.getLocalDateFrom();
        final LocalDate userRangeEnd = request.getLocalDateTo();
        final boolean withDiff = true;
        final List<PathId> userSelectedPathIds = StatisticsHelper.asPathIds(7, request.getPathId());
        final List<QueryIndicator> indicators = StatisticsHelper.asList(7, request.getIndicator());
        final Set<Integer> regions = StatisticsHelper.asSet(10, request.getRegionId());
        final RegionInclusion regionInclusion = request.getRegionInclusion();
        final QueryIndicator orderBy = request.getOrderBy();
        final OrderDirection orderDirection = request.getOrderDirection();

        int pathsCount =
                urlStatisticsService2.countPaths(request.getHostId(), specialGroup,
                        regions, regionInclusion,
                        userRangeStart, userRangeEnd, request.getDeviceType());
        PageUtils.Pager pager = PageUtils.getPage(request.getPage(), request.getPageSize(), pathsCount);
        if (pager.isEmpty()) {
            return new GetUrlListStatisticsResponse.NormalResponse(pathsCount, Collections.emptyList());
        }

        int limit;
        int startFrom;
        if (request.getPage() == 0) {
            startFrom = 0;
            limit = request.getPageSize() - userSelectedPathIds.size();
        } else {
            startFrom = pager.toRangeStart() - userSelectedPathIds.size();
            limit = request.getPageSize();
        }

        List<GetUrlListStatisticsResponse.PathStat> result =
                getStatistics(request.getHostId(), specialGroup, userRangeStart, userRangeEnd, withDiff,
                        userSelectedPathIds, indicators, regions,
                        regionInclusion, request.getDeviceType(),
                        orderBy, orderDirection, limit, startFrom);

        return new GetUrlListStatisticsResponse.NormalResponse(pathsCount, result);
    }

    @NotNull
    protected List<GetUrlListStatisticsResponse.PathStat> getStatistics(WebmasterHostId hostId,
                                                                           SpecialGroup specialGroup,
                                                                           LocalDate userRangeStart, LocalDate userRangeEnd, boolean withDiff,
                                                                           List<PathId> userSelectedPathIds,
                                                                           List<QueryIndicator> indicators,
                                                                           Set<Integer> regions, RegionInclusion regionInclusion,
                                                                           DeviceType deviceType,
                                                                           QueryIndicator orderBy, OrderDirection orderDirection,
                                                                           int limit, int startFrom) {
        List<PathId> pathIds =
                urlStatisticsService2.getPathIds(hostId, specialGroup,
                        regions, regionInclusion,
                        userRangeStart, userRangeEnd,
                        userSelectedPathIds, deviceType,
                        orderBy, orderDirection, startFrom, limit
                );
        // First page
        if (startFrom == 0 && userSelectedPathIds.size() > 0) {
            List<PathId> tmpPaths = new ArrayList<>(pathIds.size() + userSelectedPathIds.size());
            tmpPaths.addAll(userSelectedPathIds);
            tmpPaths.addAll(pathIds);
            pathIds = tmpPaths;
        }

        final RangeSet<LocalDate> rangeSet;
        if (withDiff) {
            rangeSet = RangeFactory.doubleRange(userRangeStart, userRangeEnd);
        } else {
            rangeSet = RangeFactory.singleRange(userRangeStart, userRangeEnd);
        }

        Pair<Map<PathId, String>, List<PathStat>> statistics = urlStatisticsService2.getPathStatistics(
                hostId, specialGroup, indicators,
                regionInclusion, regions,
                pathIds,
                rangeSet.span().lowerEndpoint(), rangeSet.span().upperEndpoint(), deviceType);

        Map<PathId, AccumulatorMap> accumulators = new HashMap<>();
        Function<PathId, AccumulatorMap> accumulatorFactory = q -> AccumulatorMap.create(indicators, rangeSet);
        for (PathStat pathStat : statistics.getValue()) {
            AccumulatorMap ac = accumulators.computeIfAbsent(pathStat.getPathId(), accumulatorFactory);
            ac.apply(pathStat);
        }

        List<GetUrlListStatisticsResponse.PathStat> result = new ArrayList<>();
        for (PathId id : pathIds) {
            String pathText = statistics.getKey().get(id);
            if (!StringUtils.isEmpty(pathText)) {
                AccumulatorMap accumulatorMap = accumulators.computeIfAbsent(id, accumulatorFactory);

                List<AbstractQueryStatisticsResponse.IndicatorStats> indicatorResults = new ArrayList<>();
                for (QueryIndicator indicator : indicators) {
                    List<Pair<Range<LocalDate>, Double>> indicatorStat = accumulatorMap.getIndicator(indicator);
                    List<GroupsStatisticResponse.RangeStat> rangeStats = new ArrayList<>();
                    if (withDiff) {
                        MapWithDiff.map(indicatorStat.iterator(), (r, v, d) -> {
                            rangeStats.add(new AbstractQueryStatisticsResponse.RangeStat(
                                    r.lowerEndpoint(), r.upperEndpoint(), v, d)
                            );
                        });
                    } else {
                        Pair<Range<LocalDate>, Double> range = indicatorStat.get(0);
                        rangeStats.add(new AbstractQueryStatisticsResponse.RangeStat(
                                range.getKey().lowerEndpoint(), range.getKey().upperEndpoint(), range.getValue(), null)
                        );
                    }
                    indicatorResults.add(new AbstractQueryStatisticsResponse.IndicatorStats(indicator, rangeStats));
                }

                result.add(new GetUrlListStatisticsResponse.PathStat(new ViewerPath(id, pathText), indicatorResults));
            }
        }
        return result;
    }
}
