package ru.yandex.webmaster3.viewer.http.searchurl.events;

import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.stream.Collectors;

import lombok.Setter;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.UserContext;
import ru.yandex.webmaster3.core.link.DatePoint;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.searchurl.history.SearchUrlHistoryService;
import ru.yandex.webmaster3.storage.searchurl.history.data.SearchUrlHistoryIndicator;
import ru.yandex.webmaster3.storage.searchurl.history.data.SearchUrlHistoryPoint;
import ru.yandex.webmaster3.storage.searchurl.samples.data.SearchUrlEventType;

/**
 * @author avhaliullin
 */
@Description("Получить графики истории страниц в поиске")
@ReadAction
@Category("searchurl")
public class SearchUrlEventHistoryAction extends Action<SearchUrlEventHistoryRequest, SearchUrlEventHistoryResponse> {

    @Setter
    protected SearchUrlHistoryService searchUrlHistoryService;

    @Override
    public SearchUrlEventHistoryResponse process(SearchUrlEventHistoryRequest request) throws WebmasterException {
        List<SearchUrlEventHistoryResponse.SearchUrlHistory> histories = getHistory(request);

        return new SearchUrlEventHistoryResponse(histories);
    }

    protected List<SearchUrlEventHistoryResponse.SearchUrlHistory> getHistory(SearchUrlEventHistoryRequest request)
    {
        UserContext.setUserId(request.getUserId());

        Instant dateFrom = request.getDateFrom();
        Instant dateTo = request.getDateTo();
        NavigableMap<Instant, SearchUrlHistoryPoint> rawHistory = searchUrlHistoryService.getSearchHistory(
                request.getHostId(),
                Collections.singleton(request.getNodeId()),
                EnumSet.of(SearchUrlHistoryIndicator.GONE, SearchUrlHistoryIndicator.NEW),
                dateFrom, dateTo
        );
        Map<SearchUrlEventType, TreeMap<LocalDate, Long>> indicator2History = new EnumMap<>(SearchUrlEventType.class);
        for (Map.Entry<Instant, SearchUrlHistoryPoint> entry : rawHistory.entrySet()) {
            for (SearchUrlEventType eventType : SearchUrlEventType.values()) {
                long value = entry.getValue().getValue(eventType.getIndicator(), request.getNodeId());
                indicator2History.computeIfAbsent(eventType, ign -> new TreeMap<>())
                        .put(entry.getKey().toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE).toLocalDate(), value);
            }
        }
        return indicator2History.entrySet().stream()
                .map(entry -> new SearchUrlEventHistoryResponse.SearchUrlHistory(
                        entry.getKey(),
                        entry.getValue().entrySet()
                                .stream()
                                .map(DatePoint::fromEntry)
                                .collect(Collectors.toList())
                ))
                .collect(Collectors.toList());
    }
}
