package ru.yandex.webmaster3.viewer.http.searchurl.excluded;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusEnum;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.host.HostDataState;
import ru.yandex.webmaster3.storage.host.service.HostService;
import ru.yandex.webmaster3.storage.searchurl.SearchUrlSamplesService;
import ru.yandex.webmaster3.storage.searchurl.samples.dao.ExcludedUrlConditions;
import ru.yandex.webmaster3.storage.searchurl.samples.data.SearchUrlExtendedStatus;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Condition;
import ru.yandex.webmaster3.viewer.http.AbstractUrlFilteringAction;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author avhaliullin
 */
@ReadAction
@Description("Возвращает список статусов исключенных страниц, которые присутствуют в запрошенном разделе сайта")
@Category("searchurl")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ExcludedUrlStatusesAction extends AbstractUrlFilteringAction<ExcludedUrlStatusesRequest, ExcludedUrlStatusesResponse> {
    private static final Set<SearchUrlStatusEnum> EXTENDING_STATUSES = EnumSet.of(SearchUrlStatusEnum.HTTP_ERROR);

    private final SearchUrlSamplesService searchUrlSamplesService;

    @Override
    public ExcludedUrlStatusesResponse process(ExcludedUrlStatusesRequest request) throws WebmasterException {
        Condition condition = getNodeCondition(request.getNodeId(), request.getHostId(), ExcludedUrlConditions::buildPathCondition);
        List<SearchUrlExtendedStatus> statuses = searchUrlSamplesService.getExcludedUrlExtendedStatuses(request.getHostId(), condition);
        Map<SearchUrlStatusEnum, List<Integer>> status2Codes = new EnumMap<>(SearchUrlStatusEnum.class);
        for (SearchUrlExtendedStatus exStatus : statuses) {
            List<Integer> httpCodes = status2Codes.computeIfAbsent(exStatus.getStatus(), ign -> new ArrayList<>());
            if (EXTENDING_STATUSES.contains(exStatus.getStatus())
                    && exStatus.getHttpCode() > 200
                    && exStatus.getHttpCode() < 600) {
                httpCodes.add(exStatus.getHttpCode());
            }
        }
        List<ExcludedUrlStatusesResponse.StatusInfo> result = new ArrayList<>();
        for (SearchUrlStatusEnum status : SearchUrlStatusEnum.values()) {
            List<Integer> httpCodes = status2Codes.get(status);
            if (httpCodes != null) {
                httpCodes.sort(Integer::compareTo);
                result.add(new ExcludedUrlStatusesResponse.StatusInfo(status, httpCodes));
            }
        }
        return new ExcludedUrlStatusesResponse(result);
    }
}
