package ru.yandex.webmaster3.viewer.http.sitemap;

import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestTimeout;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.sitemap.HostSitemap;
import ru.yandex.webmaster3.storage.sitemap.SitemapRecrawlRequestService;
import ru.yandex.webmaster3.storage.sitemap.dao.SitemapsCHDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.sitemap.data.RecrawlableHostSitemap;
import ru.yandex.webmaster3.viewer.http.sitemap.data.SitemapRecrawlInfo;

/**
 * @author aherman
 */
@ReadAction
@Description("Возвращает дочерние сайтмапы для индексного сайтмапа")
@RequestTimeout(17)
@Category("sitemap")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ListSitemapChildrenAction extends AbstractUserVerifiedHostAction<ListSitemapChildrenRequest, ListSitemapChildrenResponse> {

    private final SitemapRecrawlRequestService sitemapRecrawlRequestService;
    private final SitemapsCHDao sitemapsCHDao;

    @Override
    public ListSitemapChildrenResponse process(ListSitemapChildrenRequest request) {
        int pageSize = Math.max(5, request.getPageSize());
        pageSize = Math.min(25, pageSize);
        List<HostSitemap> sitemaps = sitemapsCHDao.getSitemaps(request.getHostId(), request.getParentSitemapId(), request.getFromSitemapId(), pageSize);

        Pair<Map<UUID, DateTime>, Boolean> lastRequestsInfo = sitemapRecrawlRequestService.getLastRequestsInfo(request.getHostId());
        Map<UUID, DateTime> uuidToLastRecrawlRequest = lastRequestsInfo.getLeft();
        boolean newRequestForHostIsAllowed = lastRequestsInfo.getRight();

        List<RecrawlableHostSitemap> recrawlableSitemaps = sitemaps.stream().map(sitemap -> {
            SitemapRecrawlInfo sitemapRecrawlInfo = SitemapRecrawlInfo.from(
                    sitemap.getInfo() == null ? null : sitemap.getInfo().getLastAccessDate(),
                    uuidToLastRecrawlRequest.get(sitemap.getSitemapId()),
                    newRequestForHostIsAllowed);
            return new RecrawlableHostSitemap(sitemap, sitemapRecrawlInfo);

        }).collect(Collectors.toList());

        return new ListSitemapChildrenResponse.NormalResponse(recrawlableSitemaps);
    }

}
