package ru.yandex.webmaster3.viewer.http.sitemap;

import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.sitemap.HostSitemapInfo;
import ru.yandex.webmaster3.core.util.sitemap.SitemapRecrawlQuotaUtil;
import ru.yandex.webmaster3.storage.sitemap.SitemapRecrawlRequestService;
import ru.yandex.webmaster3.storage.sitemap.dao.SitemapsCHDao;
import ru.yandex.webmaster3.core.sitemap.SitemapInfo;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.sitemap.data.HostRecrawlSitemapsLimitInfo;
import ru.yandex.webmaster3.viewer.http.sitemap.data.SitemapRecrawlInfo;

/**
 * Created by leonidrom on 22/05/2017.
 */
@Slf4j
@Description("Запросить переобход sitemap")
@WriteAction
@Category("sitemap")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class RecrawlSitemapAction extends AbstractUserVerifiedHostAction<RecrawlSitemapActionRequest, RecrawlSitemapActionResponse> {

    private final SitemapRecrawlRequestService sitemapRecrawlRequestService;
    private final SitemapsCHDao sitemapsCHDao;

    @Override
    public RecrawlSitemapActionResponse process(RecrawlSitemapActionRequest request) {
        WebmasterHostId hostId = request.getHostId();
        UUID sitemapId = request.getSitemapId();
        UUID parentId = request.getParentSitemapId();

        HostSitemapInfo sitemapInfo = sitemapsCHDao.getSitemap(hostId, parentId, sitemapId).map(SitemapInfo::toHostSitemapInfo).orElse(null);
        if (sitemapInfo == null) {
            log.info("Sitemap not found");
            return new RecrawlSitemapActionResponse.SitemapNotFoundResponse(this.getClass());
        }

        DateTime lastRequestTime = sitemapRecrawlRequestService.getLastRequestTime(hostId, sitemapId);
        SitemapRecrawlInfo recrawlInfo = SitemapRecrawlInfo.from(sitemapInfo.getLastAccessDate(), lastRequestTime, false);
        if (recrawlInfo.isPending()) {
            return new RecrawlSitemapActionResponse.RecrawlInProcessResponse(this.getClass(), recrawlInfo);
        }

        if (!sitemapRecrawlRequestService.newRequestForHostIsAllowed(request.getHostId())) {
            SitemapRecrawlQuotaUtil.QuotaUsage quotaUsage = sitemapRecrawlRequestService.getHostRecrawlQuotaInfo(request.getHostId());
            return new RecrawlSitemapActionResponse.HostRecrawlsQuotaExceededResponse(this.getClass(), quotaUsage);
        }

        log.info("Create request with host = {} and sitemapId = {}", hostId, sitemapId);
        DateTime requestDate = sitemapRecrawlRequestService.createRequest(hostId, sitemapId, parentId, sitemapInfo.getSitemapFileName());

        SitemapRecrawlQuotaUtil.QuotaUsage quotaUsage = sitemapRecrawlRequestService.getHostRecrawlQuotaInfo(request.getHostId());
        return new RecrawlSitemapActionResponse.NormalResponse(
                SitemapRecrawlInfo.newestRecrawl(requestDate),
                new HostRecrawlSitemapsLimitInfo(
                    quotaUsage.getMonthlyQuota(),
                    quotaUsage.getQuotaUsed(),
                    quotaUsage.getNearestAllowedDay()));
    }
}
