package ru.yandex.webmaster3.viewer.http.sitetree;

import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.RequestTimeout;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.sitestructure.SiteTreeNode;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.user.dao.UserSiteTreeYDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.util.URLUtil;
import ru.yandex.wmtools.common.util.uri.WebmasterUriUtils;

/**
 * <p>
 * Adds new user node to site structure.
 * <p>
 * Contract with front end:
 * <ul>
 * <li><code>nodeName</code> parameter starts with slash
 * <li>validation of <code>nodeName</code> is completely on backend site
 * <li>node name is response is provided without slash
 * </ul>
 *
 * @author azakharov
 * Date: 27.08.14
 * Time: 17:39
 */
@WriteAction
@Description(value = "Добавление пользовательского раздела в структуру сайта")
@RequestTimeout(7)
@Category("sitetree")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class AddSiteTreeNodeAction extends AbstractUserVerifiedHostAction<AddSiteTreeNodeRequest,
        AddSiteTreeNodeResponse> {
    private static final Pattern NODE_NAME_PATTERN = Pattern.compile(
            "^(/([/a-zA-Z0-9\\\\_\\-%:@&=+$,.!~*'()]+)?" +          //path
                    "(\\?([/a-zA-Z0-9\\\\_\\-%:@&=+$,.!~*'();\\?]+)?)?" +   //query
                    "(#([/a-zA-Z0-9\\\\_\\-%:@&=+$,.!~*'();\\?]+)?)?)?$"    //fragment
    );

    private final UserSiteTreeYDao userSiteTreeYDao;

    @Override
    public AddSiteTreeNodeResponse process(AddSiteTreeNodeRequest request) {
        final String originalName = cleanFragment(request.getNodeName());
        final String nodeName = getCleanNodeName(originalName);

        AddSiteTreeNodeResponse errorResponse = checkUserNodeCountAndDuplicates(request.getHostId(), nodeName);
        if (errorResponse != null) {
            return errorResponse;
        }

        errorResponse = validateNodeName(nodeName);
        if (errorResponse != null) {
            return errorResponse;
        }

        try {
            SiteTreeNode addedNode = userSiteTreeYDao.addUserNode(request.getHostId(), nodeName, originalName);
            return createResponse(addedNode);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Unable to save site tree to database",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    private AddSiteTreeNodeResponse checkUserNodeCountAndDuplicates(final WebmasterHostId hostId,
                                                                    final String nodeName) {
        try {
            List<SiteTreeNode> nodes = userSiteTreeYDao.getUserTreeNodes(hostId);
            if (nodes.size() >= SiteTreeConstants.MAX_USER_NODE_COUNT) {
                return new AddSiteTreeNodeResponse.UserNodeCountLimitExceeded(this.getClass());
            }

            for (SiteTreeNode node : nodes) {
                if (node.getName().equals(nodeName)) {
                    return new AddSiteTreeNodeResponse.DuplicationUserNodeError(this.getClass());
                }
            }
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Unable to get site tree from database",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
        return null;
    }

    private AddSiteTreeNodeResponse validateNodeName(String nodeName) {
        if (StringUtils.isEmpty(nodeName) || "/".equals(nodeName)) {
            return new AddSiteTreeNodeResponse.InvalidNodeNameResponse(this.getClass());
        }
        if (nodeName.length() > SiteTreeConstants.MAX_NODE_NAME_LENGTH) {
            return new AddSiteTreeNodeResponse.InvalidNodeNameResponse(this.getClass());
        }
        Matcher matcher = NODE_NAME_PATTERN.matcher(nodeName);
        if (!matcher.matches()) {
            return new AddSiteTreeNodeResponse.InvalidNodeNameResponse(this.getClass());
        }

        // If both path and query are absent then node name contains only slash and possibly fragment.
        // Fragment cat not be used to distinguish urls, so provided node name is considered as invalid.
        if (matcher.group(2) == null && matcher.group(4) == null) {
            return new AddSiteTreeNodeResponse.InvalidNodeNameResponse(this.getClass());
        }

        int index = nodeName.indexOf("\\");
        if (index != -1) {
            for (int i = index; i < nodeName.length() - 1; i++) {
                if (nodeName.charAt(i) == '\\') {
                    if (nodeName.charAt(i + 1) != '*') {
                        return new AddSiteTreeNodeResponse.InvalidNodeNameResponse(this.getClass());
                    }
                }
            }
            if (nodeName.charAt(nodeName.length() - 1) == '\\') {
                return new AddSiteTreeNodeResponse.InvalidNodeNameResponse(this.getClass());
            }
        }
        return null;
    }

    private String cleanFragment(String nodeName) {
        // remove fragment
        final int index = nodeName.indexOf('#');
        return (index != -1) ? nodeName.substring(0, index) : nodeName;
    }

    private String getCleanNodeName(String nodeName) {
        // workaround for converting backslash to slash: replace backslash with underscore
        // and store sequence number of newly created underscores
        final String changedUrl = nodeName.replaceAll("\\\\", "_");
        List<Integer> substNumber = new ArrayList<>();
        for (int n = 0, i = 0; i < changedUrl.length(); i++) {
            char c = changedUrl.charAt(i);
            if (c == '_') {
                n++;
                if (nodeName.charAt(i) == '\\') {
                    substNumber.add(n);
                }
            }
        }
        boolean haveLeadingSlash = nodeName.startsWith("/");

        try {
            nodeName = URLUtil.getRelativeUrl(
                    // фиг знает, что хотел сказать автор изначально - но toOldUri ожидает абсолютный путь
                    // даем ему абсолютный, чтобы потом забрать обратно относительный
                    WebmasterUriUtils.toOldUri("http://example.com" + (haveLeadingSlash ? "" : "/") + changedUrl
                    ).toURL());

            // replace stored underscores back to backslashes
            char[] string = nodeName.toCharArray();
            for (int m = 0, i = 0; i < string.length; i++) {
                if (string[i] == '_') {
                    m++;
                    if (substNumber.contains(m)) {
                        string[i] = '\\';
                    }
                }
            }
            nodeName = String.valueOf(string);
        } catch (UserException e) {
            return nodeName;
        } catch (MalformedURLException e) {
            return nodeName;
        }
        return nodeName;
    }

    private AddSiteTreeNodeResponse.NormalResponse createResponse(SiteTreeNode node) {
        String name = node.getOriginalName();
        String originalName = null;
        return new AddSiteTreeNodeResponse.NormalResponse(
                new SiteTreeNode(node.getId(),
                        node.getParentId(),
                        name,
                        originalName,
                        node.getDocCount(),
                        node.getSearchDocCount(),
                        node.getTurboDocCount(),
                        node.isUserNode(),
                        node.getSearchTurboDocs()));
    }
}
