package ru.yandex.webmaster3.viewer.http.survey;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Category;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.storage.user.CustomizableSurveyContent;
import ru.yandex.webmaster3.storage.user.CustomizableSurveyData;
import ru.yandex.webmaster3.storage.user.CustomizableSurveyTypeEnum;
import ru.yandex.webmaster3.storage.user.service.CustomizableSurveyService;
import ru.yandex.webmaster3.viewer.http.BaseWebmaster3Action;
import ru.yandex.webmaster3.viewer.http.request.AbstractFrontendRequest;

@ReadAction
@Category("notifications")
@Description("Получить попап/виджет-опрос по категории и языку")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/messages/survey/customizable")
public class GetCustomizableSurveyAction extends BaseWebmaster3Action<GetCustomizableSurveyAction.GetCustomizableSurveyRequest,
        GetCustomizableSurveyAction.GetCustomizableSurveyResponse> {

    private final CustomizableSurveyService customizableSurveyService;

    @Override
    public GetCustomizableSurveyResponse process(GetCustomizableSurveyRequest request) {
        Set<CustomizableSurveyData> serviceSurveys =
                customizableSurveyService.getServiceSurveys(request.getRouteName(), true,
                        request.getHostId(), request.getUserId());
        Optional<CustomizableSurveyData> mbPopup =
                serviceSurveys
                        .stream()
                        .filter(x -> x.getSurveyType() == CustomizableSurveyTypeEnum.POPUP)
                        .findAny();
        return mbPopup.map(GetCustomizableSurveyResponse::new)
                .orElseGet(
                        () -> new GetCustomizableSurveyResponse(serviceSurveys
                                .stream()
                                .findAny()
                                .orElse(null))
                );
    }

    @Getter
    public static class GetCustomizableSurveyRequest extends AbstractFrontendRequest {
        @Setter(onMethod_ = @RequestQueryProperty(required = true))
        private String routeName;
        @Setter(onMethod_ = @RequestQueryProperty(required = true))
        private long userId;
        @Setter(onMethod_ = @RequestQueryProperty(required = false))
        private WebmasterHostId hostId;
    }

    @Value
    public static class GetCustomizableSurveyResponse implements ActionResponse.NormalResponse {
        List<CustomizableSurveyContent> content;
        String surveyId;
        CustomizableSurveyTypeEnum surveyType;

        public GetCustomizableSurveyResponse(CustomizableSurveyData customizableSurveyData) {
            if (customizableSurveyData == null) {
                this.surveyId = null;
                this.surveyType = null;
                this.content = List.of();
            } else {
                this.surveyId = customizableSurveyData.getSurveyId();
                this.surveyType = customizableSurveyData.getSurveyType();
                this.content = customizableSurveyData.getTexts();
            }
        }
    }
}
