package ru.yandex.webmaster3.viewer.http.survey;

import java.util.Map;
import java.util.stream.Collectors;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Category;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.storage.user.SurveyState;
import ru.yandex.webmaster3.storage.user.SurveyStateEnum;
import ru.yandex.webmaster3.storage.user.dao.DashboardSurveyAnswersYDao;
import ru.yandex.webmaster3.storage.user.service.CustomizableSurveyService;
import ru.yandex.webmaster3.storage.user.service.SurveyStateService;
import ru.yandex.webmaster3.viewer.http.AbstractUserAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserRequest;

@WriteAction
@Category("notifications")
@Description("Изменить статус опросов какого-то типа для пользователя/хоста")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/messages/serviceSurvey/setStatus")
@Slf4j
public class SetSurveyStateAction extends AbstractUserAction<SetSurveyStateAction.SetSurveyStateActionRequest,
        SetSurveyStateAction.SetSurveyStateActionResponse> {

    private final SurveyStateService surveyStateService;
    private final DashboardSurveyAnswersYDao dashboardSurveyAnswersYDao;
    private final CustomizableSurveyService customizableSurveyService;

    @Override
    public SetSurveyStateActionResponse process(SetSurveyStateActionRequest request) {
        SurveyState surveyState = surveyStateService.setStatus(request.getSurveyId(),
                request.getUserId(), request.getState(), request.getHostId());
        log.info("check request: {} {} {} {} {} {}", request.getUserId(), request.getUserUUID(),
                request.getSurveyId(), request.getState(),
                request.getHostId(), request.getAnswers());
        if (request.answers != null && !request.answers.isEmpty()) {
            var answers = request
                    .answers
                    .entrySet()
                    .stream()
                    .map(entry -> {
                        var id = entry.getKey();
                        var value = entry.getValue();
                        return id + "->" + value;
                    })
                    .collect(Collectors.toList());
            var answerPlace = customizableSurveyService.getServiceSurveyAnswerPlace(request.getSurveyId());
            dashboardSurveyAnswersYDao.add(request.getUserId(), request.getHostId(), request.getSurveyId(),
                    answerPlace, answers);
        }
        return new SetSurveyStateActionResponse(surveyState);
    }

    @Getter
    public static class SetSurveyStateActionRequest extends AbstractUserRequest {
        @Setter(onMethod_ = {@Description("Id опроса"), @RequestQueryProperty(required = true)})
        private String surveyId;
        @Setter(onMethod_ = {@Description("Новое состояние"), @RequestQueryProperty(required = true)})
        private SurveyStateEnum state;
        @Setter(onMethod_ = {@Description("Хост"), @RequestQueryProperty(required = false)})
        private WebmasterHostId hostId;
        @Setter(onMethod_ = {@Description("Пользовательский ответ (если есть)"), @RequestQueryProperty(required =
                false)})
        private Map<String, String> answers;
    }

    @Value
    public static class SetSurveyStateActionResponse implements ActionResponse.NormalResponse {
        SurveyState surveyState;
    }
}
