package ru.yandex.webmaster3.viewer.http.threats;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.antispam.threats.dao.HostThreatRecheckButtonStateYDao;
import ru.yandex.webmaster3.storage.antispam.threats.dao.OwnerThreatsCHDao;
import ru.yandex.webmaster3.storage.antispam.threats.data.HostThreatInfo;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.threats.data.NonRecheckableThreat;
import ru.yandex.webmaster3.viewer.http.threats.data.RecheckableThreat;

/**
 * @author avhaliullin
 */
@ReadAction
@Description("Список угроз, обнаруженых на хосте")
@Category("threats")
@Component("/threats/list")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class GetHostThreatsAction extends AbstractUserVerifiedHostAction<GetHostThreatsRequest,
        GetHostThreatsResponse> {
    private final HostThreatRecheckButtonStateYDao hostThreatRecheckButtonStateYDao;
    private final OwnerThreatsCHDao mdbOwnerThreatsCHDao;

    @Override
    public GetHostThreatsResponse process(GetHostThreatsRequest request) {
        List<RecheckableThreat> recheckable = new ArrayList<>();
        List<NonRecheckableThreat> nonRecheckable = new ArrayList<>();
        List<HostThreatInfo> threats = mdbOwnerThreatsCHDao.getHostWithThreats(request.getHostId()).getThreats();

        Map<String, DateTime> threat2ButtonPress = new HashMap<>();
        Map<WebmasterHostId, List<HostThreatInfo>> sourceHost2Threats = threats
                .stream()
                .filter(HostThreatInfo::isAllowedToRecheck)
                .collect(Collectors.groupingBy(HostThreatInfo::getThreatSourceHost));

        for (Map.Entry<WebmasterHostId, List<HostThreatInfo>> entry : sourceHost2Threats.entrySet()) {
            Set<String> threatsForSource = entry.getValue()
                    .stream()
                    .map(HostThreatInfo::getThreat)
                    .collect(Collectors.toSet());
            threat2ButtonPress.putAll(
                    hostThreatRecheckButtonStateYDao.getRecordsForHost(entry.getKey(), threatsForSource)
            );
        }

        HostThreatUtil.classifyThreats(threats, threat2ButtonPress, recheckable, nonRecheckable);
        //  удалим дубли
        recheckable = recheckable.stream().distinct().collect(Collectors.toList());
        nonRecheckable = nonRecheckable.stream().distinct().collect(Collectors.toList());
        return new GetHostThreatsResponse(recheckable, nonRecheckable);
    }
}
