package ru.yandex.webmaster3.viewer.http.threats;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.antispam.threats.data.HostThreatInfo;
import ru.yandex.webmaster3.storage.antispam.threats.data.HostVerdictInfo;
import ru.yandex.webmaster3.storage.antispam.threats.data.VerdictUrlSample;
import ru.yandex.webmaster3.viewer.http.threats.data.HostVerdictView;
import ru.yandex.webmaster3.viewer.http.threats.data.NonRecheckableThreat;
import ru.yandex.webmaster3.viewer.http.threats.data.RecheckableThreat;
import ru.yandex.webmaster3.viewer.http.threats.data.UrlSampleView;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author avhaliullin
 */
class HostThreatUtil {
    private static UrlSampleView makeUrlSampleView(VerdictUrlSample sample, WebmasterHostId hostId) {
        return new UrlSampleView(
                IdUtils.hostIdToUrl(hostId) + sample.getUrl(),
                sample.getLastChecked(),
                sample.getInfectionChain()
        );
    }

    private static HostVerdictView makeVerdictView(HostVerdictInfo verdictInfo, WebmasterHostId hostId) {
        List<UrlSampleView> samples = verdictInfo.getVerdictUrlSamples()
                .stream()
                .map(sample -> makeUrlSampleView(sample, hostId))
                .collect(Collectors.toList());
        return new HostVerdictView(
                verdictInfo.getVerdict(),
                samples
        );
    }

    private static Pair<List<HostVerdictView>, List<UrlSampleView>> makeVerdictsView(
            Collection<HostVerdictInfo> verdictInfos, WebmasterHostId hostId) {
        List<HostVerdictView> verdictViews = new ArrayList<>();
        List<UrlSampleView> threatUrlsView = Collections.emptyList();
        for (HostVerdictInfo verdict : verdictInfos) {
            if (StringUtils.isEmpty(verdict.getVerdict())) {
                threatUrlsView = verdict.getVerdictUrlSamples()
                        .stream()
                        .map(url -> makeUrlSampleView(url, hostId))
                        .collect(Collectors.toList());
            } else {
                verdictViews.add(makeVerdictView(verdict, hostId));
            }
        }
        return Pair.of(verdictViews, threatUrlsView);
    }

    static DateTime timedLastClick(HostThreatInfo info, DateTime localClick) {
        DateTime remoteClick = info.getLastRecheck();
        // null, null
        if (remoteClick == null && localClick == null) {
            return null;
        }

        // null, X
        if (remoteClick == null) {
            return localClick;
        }

        // X, null
        if (localClick == null) {
            return remoteClick;
        }

        // X, Y
        return TimeUtils.latestOf(localClick, remoteClick);
    }

    static DateTime timedNextClick(HostThreatInfo info, DateTime lastClick) {
        if (info.getNextRecheck() != null) {// Антиспам отгрузил следующее нажатие
            if (info.getLastRecheck() != null // И предыдущее нажатие
                    && (lastClick == null || (lastClick.getMillis() / 1000 <= info.getLastRecheck().getMillis() / 1000)) //И их инфа актуальная
                    ) {
                return info.getNextRecheck();
            }

            // Теперь может случиться дуратская ситуация, когда nextClick в антиспаме высчитан по нажатию
            // на другую санкцию. Поэтому выбираем самый пессимистичный nextClick,
            // даже если lastClick из антиспама не актуальный или вообще отсутствует
            if (lastClick == null) {
                // У нас вообще не из чего высчитывать свой nextClick. Опираемся на данные антиспама
                return info.getNextRecheck();
            } else {
                // Если же есть варианты - выберем пессиместичный
                return TimeUtils.latestOf(lastClick.plus(info.getRecheckInterval()), info.getNextRecheck());
            }
        } else {
            // Антиспамной инфы про нажатие нет - опираемся на свою
            return lastClick == null ? null : lastClick.plus(info.getRecheckInterval());
        }
    }

    static void classifyThreats(Collection<HostThreatInfo> rawThreats, Map<String, DateTime> localButtonClicks,
                                List<RecheckableThreat> recheckableAcc, List<NonRecheckableThreat> nonRecheckableAcc) {
        DateTime now = DateTime.now();
        for (HostThreatInfo threatInfo : rawThreats) {
            Pair<List<HostVerdictView>, List<UrlSampleView>> samples = makeVerdictsView(
                    threatInfo.getVerdicts(),
                    threatInfo.getThreatSourceHost()
            );
            String displayName = ObjectUtils.firstNonNull(threatInfo.getDisplayName(), threatInfo.getThreat());
            if (!threatInfo.isAllowedToRecheck()) {
                nonRecheckableAcc.add(new NonRecheckableThreat(
                        threatInfo.getThreat(),
                        displayName,
                        threatInfo.getSanction(),
                        samples.getLeft(),
                        samples.getRight(),
                        null,
                        threatInfo.getThreatSourceHost(),
                        false));
                continue;
            }

            DateTime localClick = localButtonClicks.get(threatInfo.getThreat());
            switch (threatInfo.getLockMode()) {
                case TIMED:
                    DateTime lastClick = timedLastClick(threatInfo, localClick);
                    DateTime nextClick = timedNextClick(threatInfo, lastClick);
                    if (nextClick == null || now.isAfter(nextClick)) {
                        recheckableAcc.add(new RecheckableThreat(
                                threatInfo.getThreat(),
                                displayName,
                                threatInfo.getSanction(),
                                samples.getLeft(),
                                samples.getRight(),
                                now.plus(threatInfo.getRecheckInterval()),
                                threatInfo.getThreatSourceHost(),
                                null,
                                null
                        ));
                    } else {
                        nonRecheckableAcc.add(new NonRecheckableThreat(
                                threatInfo.getThreat(),
                                displayName,
                                threatInfo.getSanction(),
                                samples.getLeft(),
                                samples.getRight(),
                                nextClick, null,
                                true));
                    }
                    break;
                case UNTIL_UPDATE:
                    if (localClick == null || threatInfo.getThreatConfirmationDate().isAfter(localClick)) {
                        recheckableAcc.add(new RecheckableThreat(
                                threatInfo.getThreat(),
                                displayName,
                                threatInfo.getSanction(),
                                samples.getLeft(),
                                samples.getRight(),
                                null,
                                threatInfo.getThreatSourceHost(),
                                null,
                                null
                        ));
                    } else {
                        nonRecheckableAcc.add(new NonRecheckableThreat(
                                threatInfo.getThreat(),
                                displayName,
                                threatInfo.getSanction(),
                                samples.getLeft(),
                                samples.getRight(),
                                null, null,
                                true));
                    }
                    break;
                default:
                    throw new RuntimeException("Unknown threat lock mode: " + threatInfo.getLockMode());
            }
        }

    }
}
