package ru.yandex.webmaster3.viewer.http.turbo;

import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.model.autoparser.AutoparserToggleState;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.turbo.service.autoparser.TurboAutoparserInfoService;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.storage.user.settings.FrontendUserHostSettingsKey;
import ru.yandex.webmaster3.storage.user.settings.FrontendUserHostSettingsYDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserRequest;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: ishalaru
 * DATE: 27.09.2019
 */
@Slf4j
@ReadAction
@Category("turbo")
@Description("Возвращает список хостов для пользователя, если у хоста выключен автопарсер")
@Component("/turbo/user/popups")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class GetOneHostForPopups extends AbstractUserAction<GetOneHostForPopups.Request, GetOneHostForPopups.Response> {
    public static final String SPEED = "speed";

    private final UserHostsService userHostsService;
    private final AbtService abtService;
    private final TurboAutoparserInfoService turboAutoparserInfoService;
    private final FrontendUserHostSettingsYDao frontendUserHostSettingsYDao;

    @Override
    public Response process(Request request) {

        List<UserVerifiedHost> verifiedHosts =
                userHostsService.getVerifiedHosts(request.getWebmasterUser());
        //TODO убрать и оптимизировать, макс хостов может быть 1703
        if (verifiedHosts.size() > 25) {
            verifiedHosts = verifiedHosts.subList(0, 25);
        }
        Map<PopupType, WebmasterHostId> result = new HashMap<>(PopupType.values().length);
        final Map<WebmasterHostId, Map<FrontendUserHostSettingsKey, String>> frontendSettingsValues = frontendUserHostSettingsYDao.getValues(request.getUserId());
        autoparsedPopAp(verifiedHosts, frontendSettingsValues).ifPresent(e -> result.put(PopupType.TURBO_AUTOPARSER_POPUP, e.getWebmasterHostId()));

        return new Response.NormalResponse(result);
    }

    private Map<FrontendUserHostSettingsKey, UserVerifiedHost> search(List<UserVerifiedHost> verifiedHosts,
                                                                      Map<WebmasterHostId, Map<FrontendUserHostSettingsKey, String>> frontendKeys,
                                                                      Map<FrontendUserHostSettingsKey, String> checkingExperimentMap) {

        Map<WebmasterHostId, Set<FrontendUserHostSettingsKey>> excludedHostInfo = new HashMap<>();
        for (Map.Entry<WebmasterHostId, Map<FrontendUserHostSettingsKey, String>> frontendKeysEntry : frontendKeys.entrySet()) {
            Set<FrontendUserHostSettingsKey> excludedKeySet = new HashSet<>(checkingExperimentMap.size());
            for (FrontendUserHostSettingsKey userHostSettingsKey : checkingExperimentMap.keySet()) {
                if (frontendKeysEntry.getValue().get(userHostSettingsKey) != null) {
                    excludedKeySet.add(userHostSettingsKey);
                }
            }
            excludedHostInfo.put(frontendKeysEntry.getKey(), excludedKeySet);
        }

        // хосты, которые надо проверить на попадание в эксперименты
        List<UserVerifiedHost> hostsToCheck = verifiedHosts.stream()
                .filter(verifiedHost -> {
                    final Set<FrontendUserHostSettingsKey> frontendUserHostSettingsKeys = excludedHostInfo.get(verifiedHost.getWebmasterHostId());
                    return (frontendUserHostSettingsKeys == null || frontendUserHostSettingsKeys.size() < checkingExperimentMap.size());
                })
                .collect(Collectors.toList());

        // получим все эксперименты разом
        Map<WebmasterHostId, Map<String, String>> hostsExperiments = abtService.getHostsExperiments(
                hostsToCheck.stream().map(UserVerifiedHost::getWebmasterHostId).collect(Collectors.toList()));

        Map<FrontendUserHostSettingsKey, UserVerifiedHost> result = new HashMap<>(checkingExperimentMap.size());
        for (UserVerifiedHost verifiedHost : hostsToCheck) {
            Map<String, String> hostExperiments = hostsExperiments.get(verifiedHost.getWebmasterHostId());
            if (hostExperiments == null) {
                continue;
            }

            for (Map.Entry<FrontendUserHostSettingsKey, String> item : checkingExperimentMap.entrySet()) {
                String checkingExperiment = item.getValue();
                if (hostExperiments.containsKey(checkingExperiment)) {
                    final Set<FrontendUserHostSettingsKey> excludedSet = excludedHostInfo.get(verifiedHost.getWebmasterHostId());
                    if (excludedSet == null || !excludedSet.contains(item.getKey())) {
                        result.putIfAbsent(item.getKey(), verifiedHost);
                    }
                }
            }

            if (result.size() == checkingExperimentMap.size()) {
                break;
            }
        }
        return result;
    }

    private Optional<UserVerifiedHost> autoparsedPopAp(List<UserVerifiedHost> verifiedHosts,
                                                       Map<WebmasterHostId, Map<FrontendUserHostSettingsKey, String>> frontendKeys) {


        Optional<UserVerifiedHost> result = verifiedHosts
                .stream()
                .filter(e ->
                        frontendKeys.get(e.getWebmasterHostId()) == null ||
                                !SPEED.equals(frontendKeys.get(e.getWebmasterHostId()).
                                        get(FrontendUserHostSettingsKey.TURBO_AUTOPARSER_POPUP))
                ).filter(e -> AutoparserToggleState.OFF == turboAutoparserInfoService
                        .getAutoparseCheckBoxState(WwwUtil.cutWWWAndM(e.getWebmasterHostId()))
                )
                .findFirst();
        return result;
    }


    public enum PopupType {
        TURBO_AUTOPARSER_POPUP,
        TURBO_YML_TEASER,
        TURBO_SHOP_TEASER,
    }

    public static class Request extends AbstractUserRequest {

    }

    public static abstract class Response implements ActionResponse {

        @Value
        public static class NormalResponse extends Response implements ActionResponse.NormalResponse {
            @Description("Попап и хост для него")
            private final Map<PopupType, WebmasterHostId> result;
        }

    }
}
