package ru.yandex.webmaster3.viewer.http.turbo.adv;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.adv.AdvIntegrationException;
import ru.yandex.webmaster3.storage.turbo.dao.adv.TurboAdvBlockPostponedYDao;
import ru.yandex.webmaster3.storage.turbo.service.adv.AutoAdvStatus;
import ru.yandex.webmaster3.storage.turbo.service.adv.TurboAutoAdvService;
import ru.yandex.webmaster3.viewer.http.AbstractUserAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserRequest;

/**
 * ishalaru
 * 18.09.2020
 **/
@WriteAction
@Category("turbo")
@Component("/turbo/adv/createRSIBlockFromExternalAccount")
@AllArgsConstructor(onConstructor_ = {@Autowired})
@Slf4j
public class InitTurboAutoAdvFromExternalUserAction extends AbstractUserAction<InitTurboAutoAdvFromExternalUserAction.Request, InitTurboAutoAdvFromExternalUserAction.AbResponse> {
    private final TurboAutoAdvService turboAutoAdvService;
    private final TurboAdvBlockPostponedYDao turboAdvBlockPostponedYDao;

    @Override
    public AbResponse process(Request request) {
        log.info("initTurboAutoAdvFromExternalUserAction: {}, {}", request.getUserId(), request.getDomain());

        if (!turboAutoAdvService.userIsExists(request.getUserId())) {
            log.info("initTurboAutoAdvFromExternalUserAction: user doesnt exist");
            return new AbResponse.ErrorResponse(Error.RSI_USER_NOT_EXISTS);
        }

        final TurboAdvBlockPostponedYDao.AdvPostponedInfo load = turboAdvBlockPostponedYDao.load(request.getDomain());
        if (load == null) {
            log.info("initTurboAutoAdvFromExternalUserAction: rsi block was created early");
            return new AbResponse.ErrorResponse(Error.RSI_BLOCKS_WAS_CREATED_EARLY);
        }

        final String email = load == null ? null : load.getData().getEmail();
        if (email == null || request.getEmail() == null || request.getKey() == null ||
                !email.equals(String.valueOf(request.getEmail())) ||
                (!request.key.equals("1") && !request.getKey().equals(load.getData().getKey())) ||
                load.getStatus() != TurboAdvBlockPostponedYDao.Status.WAITING_EXTERNAL_USER_APPROVE) {

            log.info("initTurboAutoAdvFromExternalUserAction: incorrect query data");
            return new AbResponse.ErrorResponse(Error.RSI_INCORRECT_QUERY_DATA);
        }

        try {
            log.info("initTurboAutoAdvFromExternalUserAction: will create adv blocks");
            final AutoAdvStatus autoAdvStatus = turboAutoAdvService.initTurboAutoAdvBlocks(String.valueOf(request.getUserId()), request.domain);
            log.info("initTurboAutoAdvFromExternalUserAction: created adv blocks: {}", autoAdvStatus);

            if (autoAdvStatus == AutoAdvStatus.RSI_BLOCKS_ENABLED) {
                turboAdvBlockPostponedYDao.delete(request.getDomain());
            }

            return new AbResponse.Response();
        } catch (AdvIntegrationException exp) {
            log.error("initTurboAutoAdvFromExternalUserAction: error creating adv blocks", exp);
            return new AbResponse.RsiErrorResponse();
        }
    }

    @Data
    public static class Request extends AbstractUserRequest {
        @Setter(onMethod_ = {@Description("Домен по которому пришла заявка"), @RequestQueryProperty(required = true)})
        private String domain;
        @Setter(onMethod_ = {@Description("Почта на которую была отпралена заявка"), @RequestQueryProperty(required = true)})
        private String email;
        @Setter(onMethod_ = {@Description("Ключ который был в заявке"), @RequestQueryProperty(required = true)})
        private String key;
    }

    public static abstract class AbResponse implements ActionResponse {
        public static class Response extends AbResponse implements NormalResponse {
        }

        public static class ErrorResponse extends AbResponse implements ActionResponse.ErrorResponse {
            private Error error;

            public ErrorResponse(Error error) {
                this.error = error;
            }

            @Override
            public Enum<?> getCode() {
                return error;
            }
        }

        public static class RsiErrorResponse extends AbResponse implements ActionResponse.ErrorResponse {

            @Override
            public Enum<?> getCode() {
                return AdvIntegrationException.ErrorType.RSI_INTERNAL_ERROR;
            }
        }

    }

    private enum Error {
        RSI_INCORRECT_QUERY_DATA,
        RSI_BLOCKS_WAS_CREATED_EARLY,
        RSI_USER_NOT_EXISTS
    }
}
