package ru.yandex.webmaster3.viewer.http.turbo.autoparser;

import java.util.ArrayList;
import java.util.List;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId.Schema;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.TurboConstants;
import ru.yandex.webmaster3.core.turbo.model.autoparser.TurboAutoparsedHostInfo;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.turbo.dao.autoparser.TurboAutoparsedHostsSpeedsYDao;
import ru.yandex.webmaster3.storage.turbo.dao.autoparser.TurboAutoparsedHostsSpeedsYDao.OwnerSpeedInfo;
import ru.yandex.webmaster3.storage.turbo.dao.autoparser.TurboAutoparsedYDao;
import ru.yandex.webmaster3.storage.turbo.dao.autoparser.TurboAutoparserScreenshotsYDao;
import ru.yandex.webmaster3.storage.turbo.service.autoparser.TurboAutoparserInfoService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.host.verification.UsersVerifiedHostRequest;
import ru.yandex.webmaster3.viewer.http.turbo.autoparser.GetAutoparserLandingInfoAction.Request;
import ru.yandex.webmaster3.viewer.http.turbo.autoparser.GetAutoparserLandingInfoAction.Response;

/**
 * Created by Oleg Bazdyrev on 2019-07-17.
 */
@Description("Получение инфы для нового лендинга по автопарсеру с данными по скорости")
@Category("turbo")
@ReadAction
@Component("/turbo/autoparser/landing")
@RequiredArgsConstructor
public class GetAutoparserLandingInfoAction extends AbstractUserVerifiedHostAction<Request, Response> {

    private static final double DEFAULT_TURBO_SPEED = 800.0;

    private final TurboAutoparsedYDao turboAutoparsedYDao;
    private final TurboAutoparsedHostsSpeedsYDao turboAutoparsedHostsSpeedsYDao;
    private final TurboAutoparserInfoService turboAutoparserInfoService;
    private final TurboAutoparserScreenshotsYDao turboAutoparserScreenshotsYDao;
    private final HostOwnerService hostOwnerService;

    @Override
    public Response process(Request request) {
        // проверим, есть ли вообще автопарсер
        String domain = WwwUtil.cutWWWAndM(request.getHostId());
        TurboAutoparsedHostInfo autoparserInfo = turboAutoparsedYDao.get(domain);
        if (!turboAutoparserInfoService.isHostInfoActual(autoparserInfo)) {
            return new Response();
        }
        // ищем инфу по скорости
        OwnerSpeedInfo speedInfo = turboAutoparsedHostsSpeedsYDao.getSpeed(hostOwnerService.getHostOwner(domain));
        if (speedInfo == null) {
            return new Response();
        }
        // собираем примеры
        List<TurboAutoparserLandingSample> samples = new ArrayList<>();
        for (var sample : autoparserInfo.getSamples()) {
            String screenshot = turboAutoparserScreenshotsYDao.getScreenshot(sample.getOriginUrl());
            String turboScreenshot = turboAutoparserScreenshotsYDao.getScreenshot(sample.getTurboUrl() + TurboConstants.NO_REDIRECT_PARAM);
            if (screenshot != null && turboScreenshot != null) {
                samples.add(new TurboAutoparserLandingSample(
                        sample.getOriginUrl(),
                        screenshot.replaceFirst(Schema.HTTP.getSchemaPrefix(), Schema.HTTPS.getSchemaPrefix()),
                        sample.getTurboUrl(),
                        turboScreenshot.replaceFirst(Schema.HTTP.getSchemaPrefix(), Schema.HTTPS.getSchemaPrefix()),
                        sample.getTitle()));
            }
        }
        return new Response(speedInfo.getNonTurboSpeed(),
                ObjectUtils.firstNonNull(speedInfo.getTurboSpeed(), DEFAULT_TURBO_SPEED), samples);
    }

    public static final class Request extends UsersVerifiedHostRequest {

    }

    public static final class Response implements ActionResponse.NormalResponse {

        private final Double nonTurboSpeed;
        private final Double turboSpeed;
        private final List<TurboAutoparserLandingSample> samples;

        public Response() {
            this(null, null, null);
        }

        public Response(Double nonTurboSpeed, Double turboSpeed, List<TurboAutoparserLandingSample> samples) {
            this.nonTurboSpeed = nonTurboSpeed;
            this.turboSpeed = turboSpeed;
            this.samples = samples;
        }

        @Description("Средняя скорость загрузки исходных страниц (в мс)")
        public Double getNonTurboSpeed() {
            return nonTurboSpeed;
        }

        @Description("Средняя скорость загрузки турбо-страниц (в мс)")
        public Double getTurboSpeed() {
            return turboSpeed;
        }

        @Description("Примеры")
        public List<TurboAutoparserLandingSample> getSamples() {
            return samples;
        }
    }

    public static final class TurboAutoparserLandingSample {

        private final String originalUrl;
        private final String originalScreenshotUrl;
        private final String turboUrl;
        private final String turboScreenshotUrl;
        private final String title;

        public TurboAutoparserLandingSample(String originalUrl, String originalScreenshotUrl, String turboUrl,
                                            String turboSchreenshotUrl, String title) {
            this.originalUrl = originalUrl;
            this.originalScreenshotUrl = originalScreenshotUrl;
            this.turboUrl = turboUrl;
            this.turboScreenshotUrl = turboSchreenshotUrl;
            this.title = title;
        }

        @Description("Адрес исходной страницы")
        public String getOriginalUrl() {
            return originalUrl;
        }

        @Description("Скриншот исходной страницы")
        public String getOriginalScreenshotUrl() {
            return originalScreenshotUrl;
        }

        @Description("Адрес автопаршенной турбо-версии")
        public String getTurboUrl() {
            return turboUrl;
        }

        @Description("Скриншот автопаршенной турбо-версии")
        public String getTurboScreenshotUrl() {
            return turboScreenshotUrl;
        }

        @Description("Заголовок страницы")
        public String getTitle() {
            return title;
        }
    }
}
