package ru.yandex.webmaster3.viewer.http.turbo.commerce;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.Value;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedStatistics;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.turbo.dao.TurboDomainsStateCHDao;
import ru.yandex.webmaster3.storage.turbo.dao.automorda.TurboAutoMordaStatus;
import ru.yandex.webmaster3.storage.turbo.dao.commerce.model.TurboListingsInfo;
import ru.yandex.webmaster3.storage.turbo.service.TurboDomainsStateService.TurboDomainState;
import ru.yandex.webmaster3.storage.turbo.service.TurboFeedsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;
import ru.yandex.webmaster3.viewer.http.turbo.commerce.GetTurboCommerceStatusAction.Request;
import ru.yandex.webmaster3.viewer.http.turbo.commerce.GetTurboCommerceStatusAction.Response;

/**
 * Created by Oleg Bazdyrev on 2019-08-16.
 */
@ReadAction
@Description("Получение общего статуса подключения Турбо для e-commerce")
@Component("/turbo/ecommerce/status")
@Category("turbo")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class GetTurboCommerceStatusAction extends AbstractUserVerifiedHostAction<Request, Response> {

    private final TurboDomainsStateCHDao turboDomainsStateCHDao;
    private final TurboFeedsService turboFeedsService;

    @Override
    public Response process(Request request) {
        String domain = WwwUtil.cutWWWAndM(request.getHostId());
        TurboDomainState domainState = turboDomainsStateCHDao.getDomainState(domain);
        CommerceStatusEnum feedsStatus = calcFeedsStatus(domainState);
        ListingsStatusInfo listingsStatusInfo = makeListingStatusInfo(domainState);
        CommerceStatusEnum mordaStatus = calcMordaStatus(domainState);

        return new Response(
                new FeedsStatusInfo(feedsStatus),
                listingsStatusInfo,
                new MordaStatusInfo(mordaStatus));
    }


    private CommerceStatusEnum calcFeedsStatus(TurboDomainState domainState) {
        boolean hasWarnings = false;
        boolean hasOkFeeds = false;
        boolean hasFeeds = false;

        Map<TurboFeedSettings, TurboFeedStatistics> feeds = turboFeedsService.getFeedsWithStatistics(domainState.getDomain(), domainState, null);
        for (var entry : feeds.entrySet()) {
            if (!entry.getKey().isActive() || entry.getKey().getType() != TurboFeedType.YML) {
                continue;
            }
            if (entry.getValue() != null) {
                switch (entry.getValue().getState()) {
                    case OK:
                        hasOkFeeds = true;
                        break;
                    case WARNING:
                        hasWarnings = true;
                        break;
                    case ERROR:
                        return CommerceStatusEnum.ERROR;
                }
            }
            hasFeeds = true;
        }
        if (hasWarnings) {
            return CommerceStatusEnum.WARNING;
        }
        if (hasOkFeeds) {
            return CommerceStatusEnum.OK;
        }
        return hasFeeds ? CommerceStatusEnum.PROCESSING : CommerceStatusEnum.UNKNOWN;
    }

    private CommerceStatusEnum calcMordaStatus(TurboDomainState domainState) {
        TurboAutoMordaStatus mordaStatus = domainState.getAutoMordaStatus();
        if (mordaStatus == null) {
            return CommerceStatusEnum.UNKNOWN;
        }
        return mordaStatus == TurboAutoMordaStatus.OK ? CommerceStatusEnum.OK : CommerceStatusEnum.ERROR;
    }

    private ListingsStatusInfo makeListingStatusInfo(TurboDomainState domainState) {
        TurboListingsInfo turboListing = domainState.getListingsInfo();
        if (turboListing == null) {
            return new ListingsStatusInfo(CommerceStatusEnum.UNKNOWN, null);
        }
        return new ListingsStatusInfo(calcListingStatus(turboListing), calcTurboAutoListingError(turboListing));
    }

    private List<TurboListingsInfo.TurboAutoListingsError> calcTurboAutoListingError(TurboListingsInfo turboListingsInfo) {
        final List<TurboListingsInfo.TurboAutoListingsError> errors = turboListingsInfo.getErrors();
        if (errors == null) {
            return null;
        }
        return errors.stream().filter(TurboListingsInfo.TurboAutoListingsError::isVisible).collect(Collectors.toList());
    }

    private CommerceStatusEnum calcListingStatus(TurboListingsInfo turboListingsInfo) {
        switch (turboListingsInfo.getVerdict()) {
            case "OK":
                return CommerceStatusEnum.OK;
            case "PROCESSING":
                return CommerceStatusEnum.PROCESSING;
            case "WARN":
                return CommerceStatusEnum.WARNING;
            case "CRITICAL":
            case "FATAL":
                return CommerceStatusEnum.ERROR;
            default:
                return CommerceStatusEnum.UNKNOWN;
        }
    }

    public enum CommerceStatusEnum {
        OK,
        PROCESSING,
        WARNING,
        ERROR,
        UNKNOWN
    }

    public static final class Request extends AbstractUserVerifiedHostRequest {

    }

    @Value
    public static final class Response implements ActionResponse.NormalResponse {
        @Description("Статус yml-фидов")
        FeedsStatusInfo feeds;
        @Description("Статус листингов")
        ListingsStatusInfo listings;
        @Description("Статус автоморды")
        MordaStatusInfo morda;
    }

    @Value
    public static final class FeedsStatusInfo {
        CommerceStatusEnum status;
    }

    @Value
    public static final class ListingsStatusInfo {
        CommerceStatusEnum status;
        @Description("Ошибки листингов")
        List<TurboListingsInfo.TurboAutoListingsError> errors;
    }

    @Value
    public static final class MordaStatusInfo {
        CommerceStatusEnum status;
    }
}
