package ru.yandex.webmaster3.viewer.http.turbo.commerce;

import java.util.concurrent.TimeUnit;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.ValidateTurboFeedTaskData;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedState;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.turbo.dao.TurboFeedsSamplesYDao;
import ru.yandex.webmaster3.storage.turbo.service.TurboFeedsService;
import ru.yandex.webmaster3.storage.turbo.service.validation.CartUrlValidationResult;
import ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;
import ru.yandex.webmaster3.viewer.http.turbo.commerce.ValidateCartUrlAction.Request;
import ru.yandex.webmaster3.viewer.http.turbo.commerce.ValidateCartUrlAction.Response;

/**
 * Created by Oleg Bazdyrev on 07/11/2018.
 * TODO TEST
 */
@WriteAction
@Category("turbo")
@Description("Проверяет ссылку на корзину в настройках e-commerce")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ValidateCartUrlAction extends AbstractUserVerifiedHostAction<Request, Response> {

    private static final Logger log = LoggerFactory.getLogger(ValidateCartUrlAction.class);
    private static final Duration FEED_SAMPLE_MAX_AGE = Duration.standardHours(24L);
    private static final long MAX_WAIT_TIME = TimeUnit.SECONDS.toMillis(15L);

    private final TurboFeedsService turboFeedsService;
    private final TurboFeedsSamplesYDao turboFeedsSamplesYDao;
    private final TurboParserService turboParserService;
    private final WorkerClient workerClient;

    @Override
    public Response process(Request request) throws WebmasterException {
        try {
            WebmasterHostId hostId = request.getHostId();
            String domain = WwwUtil.cutWWWAndM(hostId);
            String content = null;
            Pair<String, DateTime> pair = turboFeedsSamplesYDao.getContentAndUpdateDate(domain, TurboFeedType.YML);
            if (request.forceUpdate || pair == null || pair.getRight() == null ||
                    pair.getRight().isBefore(DateTime.now().minus(FEED_SAMPLE_MAX_AGE))) { // если данных нет, или они устарели
                // найдем фид для обновления
                String ymlUrl = request.ymlUrl;
                if (Strings.isNullOrEmpty(ymlUrl)) {
                    ymlUrl = turboFeedsService.getFeeds(domain).stream()
                            .filter(f -> f.getState() != TurboFeedState.DELETED)
                            .filter(f -> f.getType() == TurboFeedType.YML).findAny()
                            .map(TurboFeedSettings::getUrl).orElse(null);
                }

                if (!Strings.isNullOrEmpty(ymlUrl)) {
                    log.info("Downloading feed {}", ymlUrl);
                    turboFeedsSamplesYDao.refreshUpdateDate(domain, TurboFeedType.YML);
                    // качаем (или обновлям)
                    long millis = System.currentTimeMillis();
                    // отсылаем задачку
                    workerClient.enqueueTask(new ValidateTurboFeedTaskData(hostId, ymlUrl, false, TurboFeedType.YML));
                    // подождем до MAX_WAIT_TIME, пока выполнится
                    do {
                        Thread.sleep(1000);
                        pair = turboFeedsSamplesYDao.getContentAndUpdateDate(domain, TurboFeedType.YML);
                        if (pair != null && pair.getRight() != null && pair.getRight().getMillis() > millis) {
                            content = pair.getLeft();
                            log.info("Downloaded feed in {} ms", System.currentTimeMillis() - millis);
                            break;
                        }
                    } while ((System.currentTimeMillis() - millis) < MAX_WAIT_TIME);
                }
            } else {
                content = pair.getLeft();
            }
            CartUrlValidationResult result;
            if (content == null) {
                result = new CartUrlValidationResult(); // не удалось найти данные для проверки
            } else {
                result = turboParserService.validateCartUrl(hostId, request.cartUrl, content);
            }

            return new Response(result);
        } catch (WebmasterYdbException e) {
            throw e.asUnchecked("Cassandra error", getClass());
        } catch (Exception e) {
            throw new WebmasterException("Other error",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Other error"), e);
        }
    }

    public static final class Request extends AbstractUserVerifiedHostRequest {

        private String ymlUrl;
        private String cartUrl;
        private boolean forceUpdate;

        @RequestQueryProperty
        @Description("Какой YML-фид использовать для проверки страницы корзины (если не указан - ищем первый попавшийся фид в настройках или берем его из кэша)")
        public void setYmlUrl(String ymlUrl) {
            this.ymlUrl = ymlUrl;
        }

        @RequestQueryProperty(required = true)
        @Description("Паттерн для ссылки на корзину")
        public void setCartUrl(String cartUrl) {
            this.cartUrl = cartUrl;
        }

        @RequestQueryProperty
        @Description("Форсировать обновление YML-фида перед проверкой страницы корзины (для дебага)")
        public void setForceUpdate(boolean forceUpdate) {
            this.forceUpdate = forceUpdate;
        }
    }

    public static final class Response implements ActionResponse.NormalResponse {

        private final CartUrlValidationResult result;

        public Response(CartUrlValidationResult result) {
            this.result = result;
        }

        @Description("Результат валидации")
        public CartUrlValidationResult getResult() {
            return result;
        }
    }
}
