package ru.yandex.webmaster3.viewer.http.turbo.feed;

import java.util.List;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.feeds.mbi.ChangeRevokeStatusException;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceSettings;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedInfo;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedStatistics;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.turbo.dao.TurboDomainsWithAcceptedOfferYDao;
import ru.yandex.webmaster3.storage.turbo.service.TurboFeedsService;
import ru.yandex.webmaster3.storage.turbo.service.settings.TurboSettingsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;
import ru.yandex.webmaster3.viewer.http.turbo.feed.ActivateTurboFeedAction.Request;

/**
 * Created by Oleg Bazdyrev on 06/07/2017.
 */
@Slf4j
@WriteAction
@Category("turbo")
@Description("Активирует указанный источник для Турбо")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class ActivateTurboFeedAction extends AbstractUserVerifiedHostAction<Request, ActivateTurboFeedResponse> {

    private final TurboSettingsService turboSettingsService;
    private final TurboFeedsService turboFeedsService;
    private final TurboDomainsWithAcceptedOfferYDao turboDomainsWithAcceptedOfferYDao;

    @Override
    public ActivateTurboFeedResponse process(Request request) {
        TurboFeedType feedType = request.feedType;
        if (feedType != TurboFeedType.RSS && feedType != TurboFeedType.YML) {
            throw new WebmasterException("Unsupported feed type " + feedType,
                    new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), "feedType", feedType.name()));
        }
        // ищем фид среди имеющихся
        String domain = WwwUtil.cutWWWAndM(request.getHostId());
        String feedUrl = request.getFeedUrl();
        TurboHostSettings turboHostSettings = null;
        final List<TurboFeedSettings> feeds = turboFeedsService.getFeeds(domain);
        boolean hasActive = false;
        int ymlActiveCount = 0;
        TurboFeedSettings feedToActivate = null;
        for (TurboFeedSettings feed : feeds) {
            if (feedUrl.equals(feed.getUrl()) && request.feedType.equals(feed.getType())) {
                feedToActivate = feed;
            }
            hasActive |= feed.isActive();
            ymlActiveCount += feed.getType() == TurboFeedType.YML && feed.isActive() ? 1 : 0;
        }
        // не нашли такой фид среди имеющихся
        if (feedToActivate == null) {
            return new ActivateTurboFeedResponse.FeedNotExistsErrorResponse();
        }
        // проверим, что фид минимум уже провалидирован
        if (!feedToActivate.getState().isExportable()) {
            return new ActivateTurboFeedResponse.NotValidatedErrorResponse();
        }

        boolean acceptOffer = turboDomainsWithAcceptedOfferYDao.contains(request.getHostId());
        acceptOffer |= request.acceptOffer;
        // проверим, что есть хотя бы один активированный фид (либо принята оферта) - в случае активации
        if (request.activate && !hasActive && !acceptOffer) {
            return new ActivateTurboFeedResponse.NeedToAcceptOfferErrorResponse();
        }
        // проверим, что задан хоть один способ заказа (при активации YML)
        if (request.feedType == TurboFeedType.YML) {
            turboHostSettings = turboSettingsService.getSettings(domain);
            TurboCommerceSettings tcs = turboHostSettings.getCommerceSettings();
            boolean orderMethodSelected = tcs != null && tcs.isOrderMethodSelected();
            if (request.activate && !orderMethodSelected) {
                return new ActivateTurboFeedResponse.InvalidCommerceSettingsResponse();
            }
            // WMC-7106 - отключаем десктоп при включении YML
            TurboDesktopSettings tds = turboSettingsService.getDesktopSettings(domain);
            if (Boolean.TRUE.equals(tds.getEnabled())) {
                turboSettingsService.updateDesktopSettings(domain,
                        new TurboDesktopSettings.TurboDesktopSettingsBuilder(tds).setEnabled(false).build());
            }
        }

        // активируем выбранный фид
        feedToActivate = feedToActivate.activate(request.activate);
        // сохраняем
        try {
            turboFeedsService.insertFeed(feedToActivate, request.getUserId());
        } catch (ChangeRevokeStatusException e) {
            log.info("feed with url - {} and domain - {} is banned", feedToActivate, domain);
            return new ActivateTurboFeedResponse.BannedFeedResponse();
        }
        turboSettingsService.notifyAboutSettingsChange(domain, turboHostSettings, null, null);
        // свежий фид
        Pair<TurboFeedSettings, TurboFeedStatistics> pair = turboFeedsService.getFeedWithStatistics(domain, feedUrl);
        return new ActivateTurboFeedResponse.NormalResponse(new TurboFeedInfo(pair.getLeft(), pair.getRight(), null));
    }

    @Getter
    public static class Request extends AbstractUserVerifiedHostRequest {

        @Description("Тип фида")
        @Setter(onMethod_ = @RequestQueryProperty(required = true))
        private TurboFeedType feedType;
        @Description("Адрес фида для активации")
        @Setter(onMethod_ = @RequestQueryProperty)
        private String feedUrl = "";
        @Description("Необходимо ли активировать фид. true - если фид надо активировать, false - если деактивировать")
        @Setter(onMethod_ = @RequestQueryProperty)
        private boolean activate = true;
        @Description("Признак принятия оферты. Если для указанного хоста еще нет активных турбо-фидов - " +
                "то необходимо значение true, иначе будет возвращена ошибка")
        @Setter(onMethod_ = @RequestQueryProperty)
        private boolean acceptOffer = false;
    }

}
