package ru.yandex.webmaster3.viewer.http.turbo.feed;

import java.util.List;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.feeds.mbi.MbiService;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.turbo.dao.app.TurboAppSettingsYDao;
import ru.yandex.webmaster3.storage.turbo.service.TurboFeedsService;
import ru.yandex.webmaster3.storage.turbo.service.settings.TurboSettingsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;
import ru.yandex.webmaster3.viewer.http.turbo.feed.RemoveTurboFeedAction.Request;

/**
 * Created by Oleg Bazdyrev on 06/07/2017.
 */
@WriteAction
@Category("turbo")
@Description("Удаление фида из настроек Турбо")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class RemoveTurboFeedAction extends AbstractUserVerifiedHostAction<Request, RemoveTurboFeedResponse> {

    private final TurboFeedsService turboFeedsService;
    private final TurboAppSettingsYDao turboAppSettingsYDao;
    private final TurboSettingsService turboSettingsService;
    private final MbiService mbiService;

    @Override
    public RemoveTurboFeedResponse process(Request request) {
        String domain = WwwUtil.cutWWWAndM(request.getHostId());
        List<TurboFeedSettings> feeds = turboFeedsService.getFeeds(domain);
        TurboFeedSettings feedToRemove = null;

        int cntActiveYmlFeeds = 0;
        for (TurboFeedSettings feed : feeds) {
            if (feed.isActive() && feed.getType().isYml()) {
                cntActiveYmlFeeds++;
            }

            if (request.feedUrl.equals(feed.getUrl())) {
                feedToRemove = feed;
            }
        }
        if (feedToRemove == null || feedToRemove.getType() == TurboFeedType.AUTO) {
            return new RemoveTurboFeedResponse.FeedNotExistsErrorResponse();
        }
        //выключаем проверку СКК если это последний включенный yml фид
        if (cntActiveYmlFeeds == 1 && feedToRemove.isActive() && feedToRemove.getType().isYml()) {
            turboAppSettingsYDao.updateSccLaunchTime(domain, null);
        }
        // удаляем
        turboFeedsService.deleteFeed(feedToRemove);
        turboSettingsService.notifyAboutSettingsChange(feedToRemove.getDomain(), null, null, null);
        return new RemoveTurboFeedResponse.NormalResponse();
    }

    public static class Request extends AbstractUserVerifiedHostRequest {

        private String feedUrl;

        public String getFeedUrl() {
            return feedUrl;
        }

        @RequestQueryProperty(required = true)
        @Description("Адрес фида для удаления")
        public void setFeedUrl(String feedUrl) {
            this.feedUrl = feedUrl;
        }
    }

}
