package ru.yandex.webmaster3.viewer.http.turbo.feed;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.ValidateTurboFeedTaskData;
import ru.yandex.webmaster3.core.turbo.model.error.TurboErrorType;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedInfo;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedState;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedStatistics;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.turbo.dao.statistics.TurboFeedsStatistics2YDao;
import ru.yandex.webmaster3.storage.turbo.service.TurboFeedsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;
import ru.yandex.webmaster3.viewer.http.turbo.feed.RevalidateTurboFeedAction.Request;

/**
 * Created by Oleg Bazdyrev on 06/07/2017.
 */
@WriteAction
@Category("turbo")
@Description("Отправляет фид на перепроверку (по xsd)")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class RevalidateTurboFeedAction extends AbstractUserVerifiedHostAction<Request, RevalidateTurboFeedResponse> {

    private final TurboFeedsService turboFeedsService;
    private final TurboFeedsStatistics2YDao turboFeedsStatistics2YDao;
    private final WMCEventsService wmcEventsService;

    private final WorkerClient workerClient;

    @Override
    public RevalidateTurboFeedResponse process(Request request) {
        try {
            String domain = WwwUtil.cutWWWAndM(request.getHostId());
            Pair<TurboFeedSettings, TurboFeedStatistics> pair = turboFeedsService.getFeedWithStatistics(domain, request.getFeedUrl());
            if (pair == null) {
                return new RevalidateTurboFeedResponse.FeedNotExistsErrorResponse();
            }
            // отправляем на перепроверку (если фид в режиме отладки)
            TurboFeedSettings feedSettings = pair.getLeft();
            if (!feedSettings.isActive()) {
                feedSettings = pair.getLeft().withState(TurboFeedState.VALIDATING).withValidateDate();
                turboFeedsService.insertFeed(feedSettings, request.getUserId());
                boolean renewHost = pair.getRight() != null && pair.getRight().getRawErrors() != null &&
                        pair.getRight().getRawErrors().stream().anyMatch(rawError ->
                                TurboErrorType.fromRawError(rawError) == TurboErrorType.FETCHER_DISALLOWED_IN_ROBOTS);
                workerClient.enqueueTask(new ValidateTurboFeedTaskData(request.getHostId(), pair.getLeft().getUrl(),
                        renewHost, feedSettings.getType()));
            }
            // удалим старую статистику, дабы не смущать пользователя устаревшими данными
            turboFeedsStatistics2YDao.deleteStatistics(request.getFeedUrl());
            return new RevalidateTurboFeedResponse.NormalResponse(new TurboFeedInfo(feedSettings, null, null));

        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Cassandra error",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    public static class Request extends AbstractUserVerifiedHostRequest {

        private String feedUrl;

        public String getFeedUrl() {
            return feedUrl;
        }

        @RequestQueryProperty(required = true)
        @Description("Адрес фида для перепроверки")
        public void setFeedUrl(String feedUrl) {
            this.feedUrl = feedUrl;
        }
    }

}
