package ru.yandex.webmaster3.viewer.http.turbo.feed;

import java.util.List;
import java.util.function.Predicate;

import com.fasterxml.jackson.annotation.JsonCreator;
import org.joda.time.DateTime;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedInfo;
import ru.yandex.webmaster3.core.turbo.model.statistics.TurboDomainStatistics;
import ru.yandex.webmaster3.storage.AbstractFilter;
import ru.yandex.webmaster3.storage.TextFilterUtil;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.ConditionFieldExtractor;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.JavaTimestampCondition;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Operator;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.TextCondition;

/**
 * Created by Oleg Bazdyrev on 31/10/2017.
 */
public class TurboFeedFilter extends AbstractFilter<TurboFeedField> {

    @JsonCreator
    public TurboFeedFilter(TurboFeedField indicator, Operation operation, String value) {
        super(indicator, operation, value);
    }

    public static Predicate<TurboFeedInfo> toPredicateTurboFeedInfo(List<TurboFeedFilter> filters) {
        Predicate<TurboFeedInfo> result = (object) -> true;
        for (TurboFeedFilter filter : filters) {
            switch (filter.getLogicalOperator()) {
                case OR:
                    result = result.or(filter.toPredicateTurboFeedInfo());
                    break;
                case AND:
                    result = result.and(filter.toPredicateTurboFeedInfo());
                    break;
                default:
                    throw new IllegalStateException("Unexpected indicator");
            }
        }
        return result;
    }

    public static Predicate<TurboDomainStatistics> toPredicateTurboDomainStatistics(List<TurboFeedFilter> filters) {
        Predicate<TurboDomainStatistics> result = (object) -> true;
        for (TurboFeedFilter filter : filters) {
            switch (filter.getLogicalOperator()) {
                case OR:
                    result = result.or(filter.toPredicateTurboDomainStatistics());
                    break;
                case AND:
                    result = result.and(filter.toPredicateTurboDomainStatistics());
                    break;
                default:
                    throw new IllegalStateException("Unexpected indicator");
            }
        }
        return result;
    }

    public Predicate<TurboDomainStatistics> toPredicateTurboDomainStatistics() {
        Operator operator = Operator.fromFilterOperation(getOperation());
        switch (getIndicator()) {
            case URL: {
                // TODO костыль для LowerCase
                Predicate<String> predicate = TextFilterUtil.getTextCondition(
                        new TurboFeedFilter(getIndicator(), getOperation(), getValue().toLowerCase()), "url")
                        .toPredicate(ConditionFieldExtractor.identity(String.class));
                return tfi -> predicate.test(tfi.getFeed().toLowerCase());
            }
            case TYPE: {
                Predicate<String> predicate = new TextCondition("type", operator, getValue()).toPredicate();
                return tfi -> predicate.test(tfi.getSource().name());
            }
        }
        return tfi -> true;
    }

    public Predicate<TurboFeedInfo> toPredicateTurboFeedInfo() {
        Operator operator = Operator.fromFilterOperation(getOperation());
        switch (getIndicator()) {
            case URL: {
                // TODO костыль для LowerCase
                Predicate<String> predicate = TextFilterUtil.getTextCondition(
                        new TurboFeedFilter(getIndicator(), getOperation(), getValue().toLowerCase()), "url")
                        .toPredicate(ConditionFieldExtractor.identity(String.class));
                return tfi -> predicate.test(tfi.getUrl().toLowerCase());
            }
            case STATE: {
                Predicate<String> predicate = new TextCondition("state", operator, getValue()).toPredicate();
                return tfi -> predicate.test(tfi.getState().name());
            }
            case TYPE: {
                Predicate<String> predicate = new TextCondition("type", operator, getValue()).toPredicate();
                return tfi -> predicate.test(tfi.getType().name());
            }
            case ADD_DATE: {
                DateTime dt = ISODateTimeFormat.dateTime().parseDateTime(getValue());
                Predicate<Long> predicate = new JavaTimestampCondition("addDate", operator, dt).toPredicate();
                return tfi -> tfi.getAddDate() != null && predicate.test(tfi.getAddDate().getMillis());
            }
            case DOWNLOAD_DATE: {
                DateTime dt = ISODateTimeFormat.dateTime().parseDateTime(getValue());
                Predicate<Long> predicate = new JavaTimestampCondition("downloadDate", operator, dt).toPredicate();
                return tfi -> tfi.getDownloadDate() != null && predicate.test(tfi.getDownloadDate().getMillis());
            }
        }
        throw new IllegalStateException("Unexpected indicator");
    }

}
