package ru.yandex.webmaster3.viewer.http.turbo.host;

import lombok.RequiredArgsConstructor;
import lombok.Value;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.checklist.data.RealTimeSiteProblemInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

import static ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum.TURBO_HOST_BAN;
import static ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum.TURBO_HOST_BAN_INFO;
import static ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum.TURBO_HOST_BAN_OK;
import static ru.yandex.webmaster3.viewer.http.turbo.host.GetTurboHostBanStatusAction.TurboHostBanStatus.BLACK;
import static ru.yandex.webmaster3.viewer.http.turbo.host.GetTurboHostBanStatusAction.TurboHostBanStatus.GREEN;
import static ru.yandex.webmaster3.viewer.http.turbo.host.GetTurboHostBanStatusAction.TurboHostBanStatus.RED;
import static ru.yandex.webmaster3.viewer.http.turbo.host.GetTurboHostBanStatusAction.TurboHostBanStatus.RED_WITH_DEADLINE;
import static ru.yandex.webmaster3.viewer.http.turbo.host.GetTurboHostBanStatusAction.TurboHostBanStatus.YELLOW;

/**
 * @author kravchenko99
 * @date 2/8/21
 */

@ReadAction
@Category("turbo")
@Description("Принята ли оферта")
@Component("/turbo/hostBan/status")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class GetTurboHostBanStatusAction extends AbstractUserVerifiedHostAction<GetTurboHostBanStatusAction.Request,
        GetTurboHostBanStatusAction.Response> {

    private final SiteProblemsService siteProblemsService;

    @Override
    public Response process(Request request) {

        RealTimeSiteProblemInfo problemInfo = siteProblemsService.getRealTimeProblemInfo(request.getHostId(),
                TURBO_HOST_BAN_OK);
        if (problemInfo != null && problemInfo.getState().isPresent()) {
            return new Response(GREEN, null);
        }

        problemInfo = siteProblemsService.getRealTimeProblemInfo(request.getHostId(), TURBO_HOST_BAN);
        if (problemInfo == null || problemInfo.getContent() == null) {
            problemInfo = siteProblemsService.getRealTimeProblemInfo(request.getHostId(), TURBO_HOST_BAN_INFO);
            if (problemInfo == null || problemInfo.getContent() == null) {
                return new Response(null, null);
            }
        }
        if (problemInfo.getState().isPresent()) {
            final SiteProblemContent.TurboHostBan problemInfoContent =
                    (SiteProblemContent.TurboHostBan) problemInfo.getContent();
            final SiteProblemContent.TurboHostBan.TurboHostBanSeverity severity = problemInfoContent.getSeverity();



            if (severity == SiteProblemContent.TurboHostBan.TurboHostBanSeverity.INFO) {
                return new Response(YELLOW, null);
            } else if (severity == SiteProblemContent.TurboHostBan.TurboHostBanSeverity.BAN) {
                return new Response(BLACK, null);
            } else if (severity == SiteProblemContent.TurboHostBan.TurboHostBanSeverity.WARNING) {
                if (problemInfoContent.getDeadlineTs() == null || problemInfoContent.getDeadlineTs() > System.currentTimeMillis()) {
                    return new Response(RED, problemInfoContent.getDeadlineTs());
                } else {
                    return new Response(RED_WITH_DEADLINE, null);
                }
            }
        }

        return new Response(null, null);
    }

    public static class Request extends AbstractUserVerifiedHostRequest {
    }

    @Value
    public static class Response implements ActionResponse.NormalResponse {
        TurboHostBanStatus status;
        Long deadlineTs;
    }

    public enum TurboHostBanStatus {
        GREEN,
        YELLOW,
        RED,
        RED_WITH_DEADLINE,
        BLACK
    }
}
