package ru.yandex.webmaster3.viewer.http.turbo.statistics;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import org.apache.commons.collections4.CollectionUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.blackbox.service.BlackboxUsersService;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingNetworkType;
import ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingPlacement;
import ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingSettings;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopSettings;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.settings.history.SettingsHistoryService;
import ru.yandex.webmaster3.storage.settings.history.SettingsHistoryType;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

import static java.lang.Integer.max;

@ReadAction
@Category("turbo")
@Description("История изменения рекламных настроек по каждому типу рекламы")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/turbo/changeAdvertisingLog/get")
public class GetTurboChangeAdvertisingLogAction extends AbstractUserVerifiedHostAction<GetTurboChangeAdvertisingLogAction.Request, GetTurboChangeAdvertisingLogAction.NormalResponse> {

    private final SettingsHistoryService settingsHistoryService;
    private final BlackboxUsersService blackboxExternalYandexUsersService;

    private final static AdvertisingPlacement[] PLACEMENTS = AdvertisingPlacement.values();
    private final static AdvertisingNetworkType[] ADVERTISING_NETWORK_TYPES = AdvertisingNetworkType.values();


    @Override
    public NormalResponse process(Request request) {
        String domain = WwwUtil.cutWWWAndM(request.getHostId());
        final Map<AdvertisingPlacement, List<AdvertisingState>> domainSettings = getPlacementsByStates(domain, SettingsHistoryType.TURBO_DOMAIN, request.getPlacements());
        final Map<AdvertisingPlacement, List<AdvertisingState>> desktopSettings = getPlacementsByStates(domain, SettingsHistoryType.TURBO_DESKTOP, request.getPlacements());

        return new NormalResponse(domainSettings, desktopSettings);
    }

    private Map<AdvertisingPlacement, List<AdvertisingState>> getPlacementsByStates(String domain, SettingsHistoryType type, List<AdvertisingPlacement> placements) {
        final List<SettingsHistoryService.LogSettingInfo> settingInfos = settingsHistoryService.turboChangesForHostAndType(domain, type);
        settingInfos.sort(Comparator.comparing(SettingsHistoryService.LogSettingInfo::getUpdateDate));
        Map<AdvertisingPlacement, List<AdvertisingSettings>> lastSettings = new HashMap<>();
        final Set<Long> userIds = new HashSet<>();
        final Map<AdvertisingPlacement, List<AdvertisingState>> placementInfos = new HashMap<>();


        for (var it : settingInfos) {
            final Map<AdvertisingPlacement, List<AdvertisingSettings>> advertisingSettings = getAdvertisingSettings(it.getNewState(), type);
            for (var placement : placements) {
                final List<AdvertisingChange> advertisingChanges = computeChanges(lastSettings.getOrDefault(placement, Collections.emptyList())
                        , advertisingSettings.getOrDefault(placement, Collections.emptyList()));
                if (!advertisingChanges.isEmpty()) {
                    lastSettings.put(placement, advertisingSettings.getOrDefault(placement, Collections.emptyList()));
                    if (it.getUserId() == -1) { // такое может быть так как я переносил текущие состояния когда делал логирования(поэтому я не могу их причислить кому нибудь из пользователей)
                        continue;
                    }
                    placementInfos.computeIfAbsent(placement, x -> new ArrayList<>()).add(new AdvertisingState(it.getUserId(), it.getUpdateDate(), advertisingChanges));
                    // -1 => состояния которые меняем внутренним сервисом и пользователи не должны видеть это
                    // 0 => изменения Яндекс.Вебмастер, т.е. показываем, что изменили мы
                    if (it.getUserId() == 0) {
                        continue;
                    }
                    userIds.add(it.getUserId());
                }

            }
        }
        final Map<Long, String> loginByUserId;
        if (userIds.isEmpty()) {
            loginByUserId = Collections.emptyMap();
        } else {
            loginByUserId = blackboxExternalYandexUsersService.mapUserIdsToLogins(userIds);
            loginByUserId.put(0L, "Yandex.Webmaster");
        }

        // проставляем логины
        for (var placementInfo : placementInfos.values()) {
            for (var state : placementInfo) {
                state.setLogin(loginByUserId.get(state.userId));
            }
            placementInfo.sort(Comparator.comparing(AdvertisingState::getUpdateDate).reversed());
        }

        return placementInfos;
    }

    private Map<AdvertisingPlacement, List<AdvertisingSettings>> getAdvertisingSettings(String state, SettingsHistoryType type) {
        if (type == SettingsHistoryType.TURBO_DOMAIN) {
            final TurboHostSettings turboHostSettings = JsonDBMapping.readValue(state, TurboHostSettings.class);
            return turboHostSettings.getAdvertisingSettings();
        } else if (type == SettingsHistoryType.TURBO_DESKTOP) {
            final TurboDesktopSettings settings = JsonDBMapping.readValue(state, TurboDesktopSettings.class);
            return settings.getAdvertisingSettings();
        } else {
            throw new IllegalStateException("expected type TURBO_DOMAIN or TURBO_DESKTOP, but actual - " + type);
        }
    }

    private List<AdvertisingChange> computeChanges(List<AdvertisingSettings> oldAdvertisingSettings, List<AdvertisingSettings> newAdvertisingSettings) {
        List<AdvertisingChange> changes = new ArrayList<>();
        Collection<AdvertisingSettings> first = CollectionUtils.subtract(oldAdvertisingSettings, newAdvertisingSettings);
        Collection<AdvertisingSettings> second = CollectionUtils.subtract(newAdvertisingSettings, oldAdvertisingSettings);

        final Map<AdvertisingNetworkType, List<AdvertisingSettings>> oldSettingsGroupingByType = first.stream().collect(Collectors.groupingBy(AdvertisingSettings::getType));
        final Map<AdvertisingNetworkType, List<AdvertisingSettings>> newSettingsGroupingByType = second.stream().collect(Collectors.groupingBy(AdvertisingSettings::getType));

        for (var type : ADVERTISING_NETWORK_TYPES) {
            changes.addAll(AdvertisingChange.computeChangesForType(
                    oldSettingsGroupingByType.getOrDefault(type, Collections.emptyList()),
                    newSettingsGroupingByType.getOrDefault(type, Collections.emptyList())));
        }

        return changes;
    }


    @Setter(onMethod_ = @RequestQueryProperty)
    public static class Request extends AbstractUserVerifiedHostRequest {
        private List<AdvertisingPlacement> placements;


        public List<AdvertisingPlacement> getPlacements() {
            if (CollectionUtils.isEmpty(placements)) {
                return Arrays.asList(PLACEMENTS);
            } else {
                return placements;
            }
        }
    }

    @Value
    public static class NormalResponse implements ActionResponse.NormalResponse {
        Map<AdvertisingPlacement, List<AdvertisingState>> domainInfo;
        Map<AdvertisingPlacement, List<AdvertisingState>> desktopInfo;
    }


    @Data
    @RequiredArgsConstructor
    private static class AdvertisingState {
        private String login;
        private final long userId;
        private final DateTime updateDate;
        private final List<AdvertisingChange> advertisingChanges;
    }

    @Value
    @AllArgsConstructor
    public static class AdvertisingChange {

        public enum Action {
            ADD, REMOVE, CHANGE
        }

        AdvertisingNetworkType type;
        String oldAlias;
        String newAlias;
        String oldValue;
        String newValue;
        ObjectNode oldData;
        ObjectNode newData;
        Action action;

        public static List<AdvertisingChange> computeChangesForType(List<AdvertisingSettings> oldSettings, List<AdvertisingSettings> newSettings) {
            var oldSettingsIt = oldSettings.iterator();
            var newSettingsIt = newSettings.iterator();

            List<AdvertisingChange> results = new ArrayList<>(max(oldSettings.size(), newSettings.size()));
            while (oldSettingsIt.hasNext() && newSettingsIt.hasNext()) {
                final AdvertisingSettings oldSetting = oldSettingsIt.next();
                final AdvertisingSettings newSetting = newSettingsIt.next();
                results.add(change(oldSetting, newSetting));
            }

            while (oldSettingsIt.hasNext()) {
                final AdvertisingSettings oldSetting = oldSettingsIt.next();
                results.add(remove(oldSetting));
            }

            while (newSettingsIt.hasNext()) {
                final AdvertisingSettings newSetting = newSettingsIt.next();
                results.add(add(newSetting));
            }
            return results;
        }

        private static AdvertisingChange change(AdvertisingSettings oldSetting, AdvertisingSettings newSetting) {
            return new AdvertisingChange(oldSetting.getType(),
                    oldSetting.getAlias(),
                    newSetting.getAlias(),
                    oldSetting.getValue(),
                    newSetting.getValue(),
                    oldSetting.getData(),
                    newSetting.getData(),
                    Action.CHANGE);
        }

        private static AdvertisingChange remove(AdvertisingSettings oldSetting) {
            return new AdvertisingChange(oldSetting.getType(),
                    oldSetting.getAlias(),
                    null,
                    oldSetting.getValue(),
                    null,
                    oldSetting.getData(),
                    null,
                    Action.REMOVE);
        }

        private static AdvertisingChange add(AdvertisingSettings newSetting) {
            return new AdvertisingChange(newSetting.getType(),
                    null,
                    newSetting.getAlias(),
                    null,
                    newSetting.getValue(),
                    null,
                    newSetting.getData(),
                    Action.ADD);
        }
    }


}
