package ru.yandex.webmaster3.viewer.http.turbo.statistics;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import org.jetbrains.annotations.NotNull;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Category;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.turbo.model.statistics.TurboClicksDayStatistics;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.searchquery.importing.dao.YtClickhouseDataLoadYDao;
import ru.yandex.webmaster3.storage.turbo.service.TurboClickStatisticsService;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

/**
 * ishalaru
 * 21.02.2020
 **/

@ReadAction
@Description(value = "Информацию о степени турбированности трафика сайта")
@Category("turbo")
@Component("/turbo/statistics/traffic/info")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TurboClickStatisticsAction extends AbstractUserVerifiedHostAction<TurboClickStatisticsAction.Request, TurboClickStatisticsAction.Response> {

    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormat.forPattern("yyyyMMdd");

    private final TurboClickStatisticsService turboClickStatisticsService;
    private final YtClickhouseDataLoadYDao ytClickhouseDataLoadYDao;

    @Override
    public Response process(Request request) {
        @NotNull String domain = WwwUtil.cutWWWAndM(request.getHostId().getPunycodeHostname());
        final List<TurboClicksDayStatistics> list = turboClickStatisticsService.list(domain, request.getStartDate(), request.getEndDate());
        LocalDate date = request.getStartDate();
        LocalDate endDate = request.getEndDate();
        if (endDate.isAfter(LocalDate.now().minusDays(1))) {
            endDate = LocalDate.now().minusDays(1);
        }
        List<TurboClicksDayStatistics> result = new ArrayList<>();
        LocalDate uploadedDate = ytClickhouseDataLoadYDao.load(YtClickhouseDataLoadType.TURBO_CLICK_DAY_STATS).getMaxProcessedDate();
        final Map<LocalDate, TurboClicksDayStatistics> map = list.stream().collect(Collectors.toMap(e -> e.getDate(), e -> e, (o, o2) -> o));
        while (!date.isAfter(endDate)) {
            if (map.containsKey(date)) {
                result.add(map.get(date));
            } else if (!date.isAfter(uploadedDate)) {
                result.add(new TurboClicksDayStatistics(domain, date, 0L, 0L, 0L, null));
            }
            date = date.plusDays(1);
        }

        return new Response(result);
    }

    public static class Request extends AbstractUserVerifiedHostRequest {
        private LocalDate startDate;
        private LocalDate endDate;

        public LocalDate getStartDate() {
            return startDate;
        }

        public LocalDate getEndDate() {
            return endDate;
        }

        @RequestQueryProperty(required = true)
        public void setStartDate(LocalDate startDate) {
            this.startDate = startDate;
        }

        @RequestQueryProperty(required = true)
        public void setEndDate(LocalDate endDate) {
            this.endDate = endDate;
        }
    }

    @Value
    public class Response implements ActionResponse.NormalResponse {
        @Getter
        List<TurboClicksDayStatistics> list;
    }
}
