package ru.yandex.webmaster3.viewer.http.url.checker2;

import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.core.util.URLEncodeUtil;
import ru.yandex.webmaster3.storage.AbstractFilter;
import ru.yandex.webmaster3.storage.TextFilterUtil;
import ru.yandex.webmaster3.storage.url.checker2.UrlCheckRequestService;
import ru.yandex.webmaster3.storage.url.checker2.UrlCheckService;
import ru.yandex.webmaster3.storage.url.common.data.UrlCheckRequestData;
import ru.yandex.webmaster3.storage.url.checker2.data.UrlCheckResultData;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.ConditionFieldExtractor;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.LongCondition;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.Operator;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.url.checker2.data.UrlCheckReport;
import ru.yandex.webmaster3.viewer.http.url.checker2.data.UrlCheckRequest;
import ru.yandex.webmaster3.viewer.http.url.checker2.data.UrlCheckRequestFilterField;
import ru.yandex.webmaster3.viewer.http.url.checker2.data.UrlCheckRequestStatus;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * Created by leonidrom on 01/03/2017.
 */
@Description("Сохраненные заявки (в процессе и выполненные)")
@ReadAction
@Category("checkurl")
public class GetUrlCheckRequestListAction extends AbstractUserVerifiedHostAction<GetUrlCheckRequestListRequest, GetUrlCheckRequestListResponse> {
    private static final Logger log = LoggerFactory.getLogger(GetUrlCheckRequestListAction.class);

    private UrlCheckRequestService urlCheckRequestService;

    @Override
    public GetUrlCheckRequestListResponse process(GetUrlCheckRequestListRequest request) {
        Predicate<UrlCheckRequestData> filterPredicate = makePredicate(request.getParameterFilters());
        List<UrlCheckRequest> requests = getRequests(request.getHostId(), filterPredicate);

        // pagination
        PageUtils.Pager pager = PageUtils.getPage(request.getP(), request.getPSize());
        List<UrlCheckRequest> resultPage = requests.stream()
                .skip(pager.toRangeStart())
                .limit(pager.getPageSize())
                .collect(Collectors.toList());

        return new GetUrlCheckRequestListResponse.NormalResponse(requests.size(), resultPage);
    }

    /**
     * Статус заявки хранится в базе результатов, поэтому мы пробегаем по
     * всем заявкам и смотрим, есть ли для нее результат. Если нет, то
     * считаем, что заявка в процессе обработки.
     */
    private List<UrlCheckRequest> getRequests(WebmasterHostId hostId, Predicate<UrlCheckRequestData> filterPredicate) {
        List<UrlCheckRequestData> requests = urlCheckRequestService.getRequests(hostId);
        List<UrlCheckResultData> results = urlCheckRequestService.getResults(hostId);
        Map<UUID, UrlCheckResultData> resultsMap = results.stream()
                .collect(Collectors.toMap(UrlCheckResultData::getRequestId, r -> r));

        List<UrlCheckRequest> list = new ArrayList<>(requests.size());
        requests.forEach(r -> {
            if (!filterPredicate.test(r)) {
                return;
            }

            UrlCheckResultData result = resultsMap.get(r.getRequestId());
            UrlCheckRequest request;
            if (result != null) {
                UrlCheckReport report = UrlCheckReport.from(r.getUrl(), result.getInfo());
                request = new UrlCheckRequest(r.getRequestId(), r.getUrl(), r.getCreatedTime(),
                        UrlCheckRequestStatus.COMPLETED, report);
            } else {
                UrlCheckRequestStatus status = UrlCheckService.isRequestExpired(r) ?
                        UrlCheckRequestStatus.COULD_NOT_CREATE : UrlCheckRequestStatus.IN_PROGRESS;
                request = new UrlCheckRequest(r.getRequestId(), r.getUrl(), r.getCreatedTime(), status);
            }

            list.add(request);
        });

        return list;
    }

    private static Predicate<UrlCheckRequestData> makePredicate(
            List<? extends AbstractFilter<UrlCheckRequestFilterField>> filters) {

        List<Predicate<UrlCheckRequestData>> predicates = new ArrayList<>();
        if (filters != null) {
            for (AbstractFilter<UrlCheckRequestFilterField> filter : filters) {
                Operator op = Operator.fromFilterOperation(filter.getOperation());
                switch (filter.getIndicator()) {
                    case URL:
                        Predicate<String> urlPredicate = TextFilterUtil.getTextCondition(filter, null)
                                .toPredicate(ConditionFieldExtractor.identity(String.class));
                        predicates.add(ucrd -> urlPredicate.test(prettifyUrl(ucrd.getUrl())));
                        break;
                    case CREATED_TIME:
                        DateTime dateTime = DateTime.parse(filter.getValue());
                        Predicate<Long> ctPredicate = new LongCondition(null, op, dateTime.getMillis()).toPredicate();
                        predicates.add(ucrd -> ctPredicate.test(ucrd.getCreatedTime().getMillis()));
                        break;
                }
            }
        }
        return predicates.stream().reduce(x -> true, Predicate::and);
    }

    private static String prettifyUrl(URL url) {
        WebmasterHostId hostId = IdUtils.urlToHostId(url);
        String path = URLEncodeUtil.prettifyUrl(url.getPath());
        return IdUtils.hostIdToReadableUrl(hostId) + path;
    }

    @Required
    public void setUrlCheckRequestService(UrlCheckRequestService urlCheckRequestService) {
        this.urlCheckRequestService = urlCheckRequestService;
    }
}
