package ru.yandex.webmaster3.viewer.http.url.checker2.data;

import org.joda.time.DateTime;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.storage.searchurl.samples.data.SearchUrlCanonicalStatus;
import ru.yandex.webmaster3.storage.searchurl.samples.data.UrlStatusInfo;
import ru.yandex.webmaster3.storage.url.checker2.data.UrlCheckInfo;

import java.net.URL;

/**
 * Created by leonidrom on 02/03/2017.
 */
public class UrlCheckReport {
    private final String title;
    private final IndexingInfo indexingInfo;
    private final SearchInfo searchInfo;
    private final UrlStatus urlStatus;

    private UrlCheckReport(String title, IndexingInfo indexingInfo, SearchInfo searchInfo, UrlStatus urlStatus) {
        this.title = title;

        this.indexingInfo = indexingInfo;
        this.searchInfo = searchInfo;
        this.urlStatus = urlStatus;
    }

    @Description("Опциональные данные об обходе роботом")
    public static class IndexingInfo {
        private final HttpCodeInfo httpCode;
        private final DateTime lastAccess;

        private IndexingInfo(HttpCodeInfo httpCode, DateTime lastAccess) {
            this.httpCode = httpCode;
            this.lastAccess = lastAccess;
        }

        @Description("Код последнего переобхода роботом")
        public HttpCodeInfo getHttpCode() {
            return httpCode;
        }

        @Description("Дата последнего переобхода роботом")
        public DateTime getLastAccess() {
            return lastAccess;
        }

        private static IndexingInfo from(UrlCheckInfo.IndexingInfo indexingInfo, UrlCheckInfo.SearchInfo searchInfo) {
            if (indexingInfo == null && searchInfo == null) {
                return null;
            }

            if (indexingInfo != null) {
                return new IndexingInfo(new HttpCodeInfo(indexingInfo.getHttpCode()),
                        indexingInfo.getLastAccess());
            } else {
                // если сведения о странице есть в поиске, то она хоть раза была обойдена роботом
                return new IndexingInfo(new HttpCodeInfo(searchInfo.getHttpCode()),
                        searchInfo.getLastAccess());
            }
        }
    }

    @Description("Статус страницы в базах обхода роботом и поиска")
    public enum UrlStatus  {
        NOT_INDEXED_AND_NOT_IN_SEARCH,
        IN_SEARCH,
        INDEXED_AND_NOT_IN_SEARCH,
        ;

        static UrlStatus from(UrlCheckInfo.IndexingInfo indexingInfo, UrlCheckInfo.SearchInfo searchInfo) {
            if (searchInfo != null) {
                // либо нет расширенного статуса (для старых данных), либо в поиске
                if (searchInfo.getExcludedStatus() == null || searchInfo.getExcludedStatus().getStatus().isSearchable()) {
                    return IN_SEARCH;
                } else {
                    return INDEXED_AND_NOT_IN_SEARCH;
                }
            }
            if (indexingInfo == null) {
                return NOT_INDEXED_AND_NOT_IN_SEARCH;
            }
            // indexingInfo != null && searchInfo == null
            return INDEXED_AND_NOT_IN_SEARCH;
        }
    }

    @Description("Опциональные данные о поиске")
    public static class SearchInfo {
        private final HttpCodeInfo httpCode;
        private final DateTime lastAccess;
        private final UrlStatusInfo excludedStatus;
        private final SearchUrlCanonicalStatus canonicalStatus;

        private SearchInfo(HttpCodeInfo httpCode, DateTime lastAccess, UrlStatusInfo excludedStatus, String url) {
            this.httpCode = httpCode;
            this.lastAccess = lastAccess;
            this.excludedStatus = excludedStatus;
            this.canonicalStatus = SearchUrlCanonicalStatus.fromUrlAndCanonical(url, excludedStatus);
        }

        @Description("Код на момент построения поисковой базы")
        public HttpCodeInfo getHttpCode() {
            return httpCode;
        }

        @Description("Дата последнего посещения на момент построения поисковой базы")
        public DateTime getLastAccess() {
            return lastAccess;
        }

        @Description("Опциональная информация для страницы, исключенной из поиска")
        public UrlStatusInfo getExcludedStatus() {
            return excludedStatus;
        }

        @Description("Статус по каноникалу")
        public SearchUrlCanonicalStatus getCanonicalStatus() {
            return canonicalStatus;
        }

        private static SearchInfo from(URL url, UrlCheckInfo.SearchInfo searchInfo) {
            // по договоренности с фронтом в случае фейковых страниц не отдаем ничего
            if (searchInfo == null || searchInfo.isFake()) {
                return null;
            }

            return new SearchInfo(new HttpCodeInfo(searchInfo.getHttpCode()),
                   searchInfo.getLastAccess(), searchInfo.getExcludedStatus(), url.toString());
        }
    }

    @Description("Заголовок страницы")
    public String getTitle() {
        return title;
    }

    @Description("Опциональные данные об обходе роботом")
    public IndexingInfo getIndexingInfo() {
        return indexingInfo;
    }

    @Description("Опциональные данные о поиске")
    public SearchInfo getSearchInfo() {
        return searchInfo;
    }

    @Description("Статус страницы в база обхода и поиска")
    public UrlStatus getUrlStatus() {
        return urlStatus;
    }

    public static UrlCheckReport from(URL url, UrlCheckInfo urlInfo) {
        UrlCheckInfo.IndexingInfo indexingInfo = urlInfo.getIndexingInfo();
        UrlCheckInfo.SearchInfo searchInfo = urlInfo.getSearchInfo();

        return new UrlCheckReport(urlInfo.getTitle(),
                IndexingInfo.from(indexingInfo, searchInfo),
                SearchInfo.from(url, searchInfo),
                UrlStatus.from(indexingInfo, searchInfo));
    }
}
