package ru.yandex.webmaster3.viewer.http.url.checker3;

import lombok.Data;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.HttpResponsePart;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.crawl.RotorSettings;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlock;
import ru.yandex.webmaster3.storage.url.checker3.data.blocks.*;
import ru.yandex.webmaster3.viewer.http.url.checker3.data.PageContentResponseDataBlock;
import ru.yandex.webmaster3.viewer.http.url.checker3.data.SearchInfoResponseDataBlock;

import java.util.List;

/**
 * @author leonidrom
 */
public class GetUrlCheckInfoNormalResponse extends GetUrlCheckInfoResponse implements ActionResponse.NormalResponse {
    private final SearchBaseInfo searchBaseInfo;
    private final UrlCheckInfo urlCheckInfo;

    @Getter
    private final boolean inProgress;

    public GetUrlCheckInfoNormalResponse() {
        this.urlCheckInfo = null;
        this.searchBaseInfo = null;
        this.inProgress = true;
    }

    public GetUrlCheckInfoNormalResponse(UrlCheckInfo urlCheckInfo, SearchBaseInfo searchBaseInfo, boolean inProgress) {
        this.urlCheckInfo = urlCheckInfo;
        this.searchBaseInfo = searchBaseInfo;
        this.inProgress = inProgress;
    }

    @RequiredArgsConstructor
    public static class UrlCheckInfo {
        private final RotorSettings.RenderSettings currentRenderSettings;
        private final RotorSettings.RenderSettings usedRenderSettings;

        private final UrlCheckInfo.PageContent pageContent;
        private final HttpResponsePart technicalInfo;
        private final List<ServerResponseData.ResourceInfo> resourceInfos;

        @Description("Все про контент страницы")
        public UrlCheckInfo.PageContent getPageContent() {
            return pageContent;
        }

        @Description("Все про ответ сервера")
        public HttpResponsePart getTechnicalInfo() {
            return technicalInfo;
        }

        @Description("Текущие настройки исполнения JS")
        public RotorSettings.RenderSettings getCurrentRenderSettings() {
            return currentRenderSettings;
        }

        @Description("Использованные настройки исполнения JS")
        public RotorSettings.RenderSettings getUsedRenderSettings() {
            return usedRenderSettings;
        }

        @Description("Информация о ресурсах")
        public List<ServerResponseData.ResourceInfo> getResourceInfos() {
            return resourceInfos;
        }

        @RequiredArgsConstructor
        @Description("Контент страницы")
        public static class PageContent {
            private final String title;
            private final String description;
            private final String base64PNGScreenshot;
            private final String content;

            @Description("Title страницы")
            public String getTitle() {
                return title;
            }

            @Description("Description страницы")
            public String getDescription() {
                return description;
            }

            @Description("Скриншот страницы в формате base64 png")
            public String getBase64PNGScreenshot() {
                return base64PNGScreenshot;
            }

            @Description("Текстовый контент страницы")
            public String getContent() {
                return content;
            }
        }

        public static UrlCheckInfo fromDataBlocks(
                RotorSettings.RenderSettings currentRenderSettings,
                UrlCheckDataBlock rotorRenderSettingsDataBlock,
                UrlCheckDataBlock rotorCheckDataBlock,
                UrlCheckDataBlock pageDataBlock,
                UrlCheckDataBlock serverResponseDataBlock) {
            var rotorCheckData = JsonMapping.readValue(rotorCheckDataBlock.getData(), RotorCheckData.class);
            var pageData = JsonMapping.readValue(pageDataBlock.getData(), PageTextContentTitleDescData.class);
            var serverResponseData = JsonMapping.readValue(serverResponseDataBlock.getData(), ServerResponseData.class);
            var rotorRenderSettingsData = JsonMapping.readValue(rotorRenderSettingsDataBlock.getData(),
                    RotorRenderSettingsData.class);

            UrlCheckInfo.PageContent pageContent = new UrlCheckInfo.PageContent(pageData.getTitle(), pageData.getDescription(),
                    rotorCheckData.getBase64Screenshot(), pageData.getText());
            return new UrlCheckInfo(currentRenderSettings, rotorRenderSettingsData.getRenderSettings(), pageContent,
                    serverResponseData.getHttpResponsePart(), serverResponseData.getResourceInfos());
        }
    }

    @Data
    public static class SearchBaseInfo {
        private final RotorSettings.RenderSettings currentRenderSettings;
        private final DateTime searchBaseDate;
        private final SearchInfoResponseDataBlock searchInfoData;
        private final PageContentResponseDataBlock pageTextContentData;

        @Description("Дата поисковой базы")
        public DateTime getSearchBaseDate() {
            return searchBaseDate;
        }

        public SearchInfoResponseDataBlock getSearchInfoData() {
            return searchInfoData;
        }

        public PageContentResponseDataBlock getPageTextContentData() {
            return pageTextContentData;
        }

        public static SearchBaseInfo fromDataBlocks(
                String url,
                RotorSettings.RenderSettings currentRenderSettings,
                UrlCheckDataBlock searchInfoDataBlock,
                UrlCheckDataBlock indexingInfoDataBlock,
                UrlCheckDataBlock pageContentDataBlock,
                UrlCheckDataBlock searchBaseDataBlock) {

            var searchBaseDateData = JsonMapping.readValue(searchBaseDataBlock.getData(), SearchBaseDateData.class);
            var pageContentData = PageContentResponseDataBlock.fromDataBlock(pageContentDataBlock);
            var searchInfoResponseData = SearchInfoResponseDataBlock.fromDataBlocks(url, searchInfoDataBlock, indexingInfoDataBlock);

            return new SearchBaseInfo(
                    currentRenderSettings,
                    searchBaseDateData.getSearchBaseDate(),
                    searchInfoResponseData,
                    pageContentData
            );
        }
    }

    @Description("Результат проверки")
    public UrlCheckInfo getUrlCheckInfo() {
        return urlCheckInfo;
    }

    @Description("Версия страницы в базе")
    public SearchBaseInfo getSearchBaseInfo() {
        return searchBaseInfo;
    }

    public static GetUrlCheckInfoNormalResponse fromDataBlocks(
            String url,
            RotorSettings.RenderSettings currentRenderSettings,
            UrlCheckDataBlock searchInfoDataBlock,
            UrlCheckDataBlock indexingInfoDataBlock,
            UrlCheckDataBlock pageContentDataBlock,
            UrlCheckDataBlock searchBaseDateDataBlock,
            UrlCheckDataBlock rotorRenderSettingsDataBlock,
            UrlCheckDataBlock rotorCheckDataBlock,
            UrlCheckDataBlock pageDataBlock,
            UrlCheckDataBlock serverResponseDataBlock) {

        SearchBaseInfo searchBaseInfo = SearchBaseInfo.fromDataBlocks(url, currentRenderSettings, searchInfoDataBlock,
                indexingInfoDataBlock, pageContentDataBlock, searchBaseDateDataBlock);

        var urlCheckInfo = UrlCheckInfo.fromDataBlocks(currentRenderSettings, rotorRenderSettingsDataBlock,
                rotorCheckDataBlock, pageDataBlock, serverResponseDataBlock);

        return new GetUrlCheckInfoNormalResponse(urlCheckInfo, searchBaseInfo, false);
    }
}
