package ru.yandex.webmaster3.viewer.http.url.checker3;

import NUrlChecker.Request;
import com.datastax.driver.core.utils.UUIDs;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.logbroker.writer.LogbrokerClient;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.url.checker3.UrlCheckDeviceType;
import ru.yandex.webmaster3.core.url.checker3.UrlCheckRequestParams;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.service.DisplayNameService2;
import ru.yandex.webmaster3.storage.jupiter.JupiterUtils;
import ru.yandex.webmaster3.storage.url.checker3.UrlCheckRequestService2;
import ru.yandex.webmaster3.storage.url.checker3.dao.UrlCheckDataBlocksYDao;
import ru.yandex.webmaster3.storage.url.checker3.dao.UrlCheckRequests2YDao;
import ru.yandex.webmaster3.storage.url.checker3.data.FetchUrlCheckDataBlockTaskData;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockType;
import ru.yandex.webmaster3.storage.url.checker3.data.blocks.SearchBaseDateData;
import ru.yandex.webmaster3.storage.url.checker3.service.UrlCheckDataBlocksService;
import ru.yandex.webmaster3.storage.url.common.data.UrlCheckRequestSource;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.common.response.CommonHostInfo;
import ru.yandex.webmaster3.viewer.http.url.checker2.AddUrlCheckResponse;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.util.uri.WebmasterUriUtils;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.List;
import java.util.UUID;

/**
 * @author leonidrom
 */
@Description("Запросить проверку урла")
@WriteAction
@Category("crawl")
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/urlchecker3/start")
public class StartUrlCheckAction extends AbstractUserVerifiedHostAction<StartUrlCheckRequest, StartUrlCheckResponse> {
    private final HostOwnerService hostOwnerService;
    private final UrlCheckRequestService2 urlCheckRequestService2;
    private final UrlCheckDataBlocksService urlCheckDataBlocksService;
    private final UserHostsService userHostsService;
    private final DisplayNameService2 displayNameService2;

    @Override
    public StartUrlCheckResponse process(StartUrlCheckRequest request) {
        UrlCheckRequestParams urlCheckParams = request.getRequestParams();

        int requestsLeft = urlCheckRequestService2.getRequestsLeftForHostToday(request.getHostId());
        if (requestsLeft == 0) {
            return new StartUrlCheckResponse.HostLimitReachedResponse();
        }

        URL url;
        try {
            url = processUrl(request.getHostId(), urlCheckParams.getUrl());
        } catch (Exception e) {
            return new StartUrlCheckResponse.InvalidUrlResponse();
        }

        WebmasterHostId hostOfUrl = IdUtils.urlToHostId(url);
        if (!compareHosts(request.getHostId(), hostOfUrl)) {
            UserVerifiedHost userHost = userHostsService.getVerifiedHost(request.getWebmasterUser(), hostOfUrl);
            if (userHost != null) {
                // another host is verified by user
                CommonHostInfo anotherHostInfo = getCommonHostInfo(hostOfUrl);
                return new StartUrlCheckResponse.AddToAnotherHostResponse(anotherHostInfo);
            } else {
                // another host is not verified by user
                return new StartUrlCheckResponse.UrlIsNotFromHostResponse();
            }
        }

        urlCheckParams = urlCheckParams.withUrl(url.toExternalForm()); // сделаем урл абсолютным
        var requestId = urlCheckDataBlocksService.sendUrlCheckRequest(urlCheckParams);

        return new StartUrlCheckResponse.NormalResponse(requestId);
    }

    private URL processUrl(WebmasterHostId hostId, String urlS) throws UserException, MalformedURLException {
        URI uri;
        if (urlS.startsWith("/")) {
            uri = WebmasterUriUtils.toOldUri(IdUtils.hostIdToUrl(hostId) + urlS);
        } else {
            uri = WebmasterUriUtils.toOldUri(urlS);
        }

        if (StringUtils.isEmpty(uri.getPath())) {
            uri = uri.resolve("/");
        }

        return uri.toURL();
    }

    private boolean compareHosts(WebmasterHostId expected, WebmasterHostId actual) {
        if (expected.equals(actual)) {
            return true;
        }

        String expectedHostPart = expected.getPunycodeHostname();
        String actualHostPart = actual.getPunycodeHostname();

        return expected.getPort() == actual.getPort() && WwwUtil.equalsIgnoreWww(actualHostPart, expectedHostPart);
    }

    private CommonHostInfo getCommonHostInfo(WebmasterHostId hostId) {
        String hostUrlWithDisplayName = displayNameService2.getHostUrlWithDisplayName(hostId);

        return new CommonHostInfo(hostId, hostUrlWithDisplayName, false, null);
    }
}
