package ru.yandex.webmaster3.viewer.http.user;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemStorageType;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.concurrency.AsyncCtx;
import ru.yandex.webmaster3.core.concurrency.AsyncTask;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestTimeout;
import ru.yandex.webmaster3.core.iks.data.Sqi;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.metrika.counters.MetrikaCountersUtil;
import ru.yandex.webmaster3.core.turbo.model.autoparser.AutoparserToggleState;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.core.util.ProfilingWatch;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.checklist.data.SummarySiteProblemsInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.host.HostDataState;
import ru.yandex.webmaster3.storage.host.HostIndicatorsDiff;
import ru.yandex.webmaster3.storage.host.HostIndicatorsState;
import ru.yandex.webmaster3.storage.host.moderation.camelcase.service.DisplayNameService2;
import ru.yandex.webmaster3.storage.iks.IksService;
import ru.yandex.webmaster3.storage.metrika.MetrikaCounterBindingService;
import ru.yandex.webmaster3.storage.metrika.MetrikaCrawlStateService;
import ru.yandex.webmaster3.storage.metrika.data.MetrikaDomainCrawlState;
import ru.yandex.webmaster3.storage.sitestructure.SiteStructureService;
import ru.yandex.webmaster3.storage.turbo.service.TurboSearchUrlsStatisticsService;
import ru.yandex.webmaster3.storage.turbo.service.TurboSearchUrlsStatisticsService.TurboDomainStats;
import ru.yandex.webmaster3.storage.user.UserUnverifiedHost;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.storage.util.HostListTreeBuilder;
import ru.yandex.webmaster3.storage.util.HostListTreeBuilder.HostCounters;
import ru.yandex.webmaster3.storage.util.HostListTreeBuilder.HostTreeNode;
import ru.yandex.webmaster3.viewer.http.common.response.CommonHostInfo;
import ru.yandex.webmaster3.viewer.http.common.response.HostDataStatus;
import ru.yandex.webmaster3.viewer.http.common.response.HostHealth;
import ru.yandex.webmaster3.viewer.http.concurrency.AsyncAction;
import ru.yandex.webmaster3.viewer.http.user.GetHostListResponse.GrowingNumber;
import ru.yandex.webmaster3.viewer.http.user.GetHostListResponse.HostTurboInfo;
import ru.yandex.webmaster3.viewer.http.user.GetHostListResponse.ShortHostInfo;
import ru.yandex.webmaster3.viewer.http.user.GetHostListResponse.ShortHostInfo.ShortHostInfoBuilder;
import ru.yandex.webmaster3.viewer.util.HostnameFilter;

/**
 * @author aherman
 */
@Slf4j
@ReadAction
@Category("user")
@Description(value = "Выдает список добавленных хостов пользователя")
@RequestTimeout(194)
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class GetHostListAction extends AsyncAction<GetHostListRequest, GetHostListResponse> {

    private final AbtService abtService;
    private final DisplayNameService2 displayNameService2;
    private final HostListTreeBuilder hostListTreeBuilder;
    private final HostOwnerService hostOwnerService;
    private final IksService iksService;
    private final MetrikaCounterBindingService metrikaCounterBindingService;
    private final MetrikaCrawlStateService metrikaCrawlStateService;
    private final SiteProblemsService siteProblemsService;
    private final SiteStructureService siteStructureService;
    private final TurboSearchUrlsStatisticsService turboSearchUrlsStatisticsService;
    private final UserHostsService userHostsService;
    private final FeedsService feedsService;

    @Override
    public GetHostListResponse processAsync(AsyncCtx ctx, GetHostListRequest request) {
        ProfilingWatch watch = new ProfilingWatch(Duration.ofMillis(150));
        try {
            WebmasterUser user = request.getWebmasterUser();
            PageUtils.Pager page = PageUtils.getPage(request.getPage(), request.getPageSize());
            HostnameFilter hostnameFilter = HostnameFilter.create(request.getHostnameFilter());
            if (hostnameFilter == null) {
                return new GetHostListResponse.NormalResponse(0, 0, 0, 0, Collections.emptyList(), Collections.emptyMap());
            }
            AsyncTask<List<UserVerifiedHost>> verifiedHostsTask = ctx.fork(() -> userHostsService.getVerifiedHosts(user));
            AsyncTask<List<UserUnverifiedHost>> unverifiedHostsTask = ctx.fork(() -> userHostsService.getMayBeUnverifiedHosts(user));
            AsyncTask<List<WebmasterHostId>> pinnedHostsTask = ctx.fork(() -> userHostsService.getPinnedHosts(user));

            watch.mark("fetchMirrorsGen");

            List<WebmasterHostId> pinnedHosts = pinnedHostsTask.join();
            List<UserUnverifiedHost> unverifiedHosts = unverifiedHostsTask.join();
            List<UserVerifiedHost> verifiedHosts = verifiedHostsTask.join();
            watch.mark("fetchHosts");

            Pair<List<HostTreeNode>, HostCounters> treeResults =
                    hostListTreeBuilder.createHostTree(verifiedHosts, unverifiedHosts, pinnedHosts);
            watch.mark("treeBuilder");

            List<HostTreeNode> mainMirrors = treeResults.getLeft();
            var hostsWithoutVerifiedMainMirror = getHostsWithoutVerifiedMainMirror(mainMirrors);

            mainMirrors = filterByName(mainMirrors, hostnameFilter);

            if (mainMirrors.isEmpty()) {
                return new GetHostListResponse.NormalResponse(verifiedHosts.size(), unverifiedHosts.size(), mainMirrors.size(),
                        0, Collections.emptyList(), hostsWithoutVerifiedMainMirror);
            }


            int rangeStart = Math.min(page.toRangeStart(), mainMirrors.size() - 1);
            int rangeEnd = Math.min(rangeStart + page.getPageSize(), mainMirrors.size());
            List<HostTreeNode> mirrorsOnPage = mainMirrors.subList(rangeStart, rangeEnd);

            Set<WebmasterHostId> hostsToDisplay = new HashSet<>();
            for (HostTreeNode node : mirrorsOnPage) {
                if (node.isVerified()) {
                    hostsToDisplay.add(node.getHostId());
                }
                for (HostTreeNode mirror : node.getChildren()) {
                    if (mirror.isVerified()) {
                        hostsToDisplay.add(mirror.getHostId());
                    }
                }
            }

            List<AsyncTask<HostDataState>> hostIdTasks = hostsToDisplay.stream()
                    .map(hostId -> ctx.fork(() -> siteStructureService.getHostDataState(hostId)))
                    .collect(Collectors.toList());

            AsyncTask<Map<WebmasterHostId, SummarySiteProblemsInfo>> rtProblemTask =
                    ctx.fork(() -> siteProblemsService.getRealTimeProblemsSummary(ctx, hostsToDisplay));

            // соберем домены для инфы о турбо
            Set<String> domainsToDisplay = hostsToDisplay.stream().map(WwwUtil::cutWWWAndM).collect(Collectors.toSet());
            // оунеры для ИКСа
            Set<String> ownersToDisplay = domainsToDisplay.stream().map(hostOwnerService::getMascotHostOwner)
                    .collect(Collectors.toSet());
            AsyncTask<Map<String, Sqi>> sqiTask = ctx.fork(() -> iksService.getSqiForOwners(ctx, ownersToDisplay));

            AsyncTask<Map<String, TurboDomainStats>> turboHostInfoTask = ctx.fork(() -> turboSearchUrlsStatisticsService.getTurboInfos(domainsToDisplay));

            AsyncTask<Map<WebmasterHostId, Map<String, String>>> hostsExperimentsTask = ctx.fork(() -> abtService.getHostsExperiments(hostsToDisplay));

            Set<String> metrikaDomainsToDisplay = hostsToDisplay.stream().map(MetrikaCountersUtil::hostToPunycodeDomain).collect(Collectors.toSet());
            AsyncTask<Map<String, MetrikaDomainCrawlState>> domainsCrawlStateTask = ctx.fork(() -> metrikaCrawlStateService.getDomainsCrawlState(
                    metrikaDomainsToDisplay, metrikaCounterBindingService.getBindingsForDomains(metrikaDomainsToDisplay, true)));

            AsyncTask<Set<String>> domainsWithSuccessfulGoodsFeedsTask =
                    ctx.fork(() -> feedsService.getDomainsWithTvInSearch(new ArrayList<>(domainsToDisplay)));

            // Авторитетные источники предлагают join'ить таски в порядке, противоположном их fork'у
            Collections.reverse(hostIdTasks);

            Map<String, Sqi> owner2Sqi = sqiTask.join();
            watch.mark("fetchSqi");

            Map<WebmasterHostId, SummarySiteProblemsInfo> rtProblems = rtProblemTask.join();
            watch.mark("fetchRTProblems");

            Map<WebmasterHostId, HostDataState> hostId2DataState = hostIdTasks.stream()
                    .map(AsyncTask::join)
                    .filter(Objects::nonNull)
                    .collect(Collectors.toMap(HostDataState::getHostId, Function.identity()));
            watch.mark("fetchHDS");

            Map<String, TurboDomainStats> domain2TurboHostInfoState = turboHostInfoTask.join();
            watch.mark("fetchTurboHostInfoTask");

            Map<WebmasterHostId, Map<String, String>> host2Experiments = hostsExperimentsTask.join();
            watch.mark("fetchExperiments");

            Map<String, MetrikaDomainCrawlState> domainsCrawlState = domainsCrawlStateTask.join();
            Set<String> domainsWithSuccessfulGoodsFeeds = domainsWithSuccessfulGoodsFeedsTask.join();

            List<ShortHostInfo> result = new ArrayList<>();

            for (HostTreeNode node : mirrorsOnPage) {
                WebmasterHostId hostId = node.getHostId();
                String domain = WwwUtil.cutWWWAndM(hostId);
                String metrikaDomain = MetrikaCountersUtil.hostToPunycodeDomain(hostId);
                String owner = hostOwnerService.getMascotHostOwner(domain);
                final TurboDomainStats turboStats = domain2TurboHostInfoState.getOrDefault(domain, TurboDomainStats.EMPTY);

                ShortHostInfo info = toResponseInfo(node,
                        false,
                        rtProblems.get(hostId),
                        hostId2DataState.get(hostId),
                        owner2Sqi.get(owner),
                        turboStats,
                        domainsCrawlState.getOrDefault(metrikaDomain, new MetrikaDomainCrawlState(metrikaDomain)),
                        domainsWithSuccessfulGoodsFeeds.contains(domain)
                );

                if (!node.getChildren().isEmpty()) {
                    for (HostTreeNode mirror : node.getChildren()) {
                        WebmasterHostId mirrorHostId = mirror.getHostId();
                        String mirrorDomain = WwwUtil.cutWWWAndM(mirrorHostId);
                        String mirrorMetrikaDomain = MetrikaCountersUtil.hostToPunycodeDomain(hostId);
                        String mirrorOwner = hostOwnerService.getMascotHostOwner(mirrorDomain);
                        TurboDomainStats mirrorStats = domain2TurboHostInfoState.getOrDefault(mirrorDomain, TurboDomainStats.EMPTY);
                        ShortHostInfo mirrorInfo = toResponseInfo(mirror,
                                false,
                                rtProblems.get(mirrorHostId),
                                hostId2DataState.get(mirrorHostId),
                                owner2Sqi.get(mirrorOwner),
                                mirrorStats,
                                domainsCrawlState.getOrDefault(mirrorMetrikaDomain, new MetrikaDomainCrawlState(mirrorMetrikaDomain)),
                                domainsWithSuccessfulGoodsFeeds.contains(domain)
                        );
                        info.getMirrors().add(mirrorInfo);
                    }
                }

                result.add(info);
            }

            HostCounters hostCounters = treeResults.getRight();
            watch.mark("gatherResponse");

            return new GetHostListResponse.NormalResponse(hostCounters.getVerifiedHostsCount(),
                    hostCounters.getUnverifiedHostsCount(), hostCounters.getMainMirrorHostsCount(),
                    mainMirrors.size(), result, hostsWithoutVerifiedMainMirror);
        } finally {
            log.info("Host list profiling: {}", watch);
        }
    }

    private Map<WebmasterHostId, WebmasterHostId> getHostsWithoutVerifiedMainMirror(List<HostTreeNode> treeNodes) {
        return treeNodes.stream()
                // подтвержденно ли главное зеркало
                .filter(x -> !x.isVerified())
                .flatMap(parent -> parent.getChildren().stream().filter(HostTreeNode::isVerified).map(child -> Pair.of(child.getHostId(), parent.getHostId())))
                .collect(W3Collectors.toTreeMap());

    }

    private List<HostTreeNode> filterByName(List<HostTreeNode> mainMirrors,
                                            HostnameFilter hostnameFilter) {
        List<HostTreeNode> result = new ArrayList<>();
        for (HostTreeNode mainMirror : mainMirrors) {
            if (hostnameFilter.accept(mainMirror.getHostId())) {
                result.add(mainMirror);
            } else {
                for (HostTreeNode mirror : mainMirror.getChildren()) {
                    if (hostnameFilter.accept(mirror.getHostId())) {
                        result.add(mainMirror);
                        break;
                    }
                }
            }
        }
        return result;
    }

    private ShortHostInfo toResponseInfo(HostTreeNode node,
                                         boolean isInMarket,
                                         SummarySiteProblemsInfo rtProblems,
                                         HostDataState dataState, Sqi sqi,
                                         TurboDomainStats domainStats,
                                         MetrikaDomainCrawlState crawlState,
                                         boolean hasSuccessfulGoodFeeds) {
        WebmasterHostId hostId = node.getHostId();
        Pair<Boolean, String> ownerInfo = IksService.comparePunycodeHostNameWithPunycodeOwnerName(
                hostId, Optional.ofNullable(sqi).map(Sqi::getOwner).orElse(hostId.getPunycodeHostname()));
        if (sqi == null) {
            sqi = IksService.defaultSqi("");
        }
        String displayName = displayNameService2.getHostUrlWithDisplayName(hostId);

        return toResponseInfo(hostId, isInMarket, node.isAdded(), node.isVerified(), node.isPinned(), dataState,
                rtProblems, node.getPinnedNodes(), ownerInfo, sqi, domainStats, crawlState, displayName, hasSuccessfulGoodFeeds);
    }

    public static ShortHostInfo toResponseInfo(WebmasterHostId hostId, boolean isInMarket,
                                                      boolean added, boolean verified, boolean pinned,
                                                      HostDataState dataState,
                                                      SummarySiteProblemsInfo rtProblems,
                                                      List<WebmasterHostId> pinnedNodes,
                                                      Pair<Boolean, String> ownerInfo,
                                                      Sqi sqi,
                                                      TurboDomainStats ds,
                                                      MetrikaDomainCrawlState crawlState,
                                                      String displayName,
                                                      boolean hasSuccessfulGoodFeeds) {
        GetHostListResponse.HostData hostData = null;
        if (verified) {
            if (dataState != null && dataState.getHostIndicatorsDiff() != null) {
                HostIndicatorsDiff diff = dataState.getHostIndicatorsDiff();
                hostData = new GetHostListResponse.HostData(
                        GrowingNumber.create(sqi.getPreviousIks(), sqi.getIks()),
                        GrowingNumber.create(diff, HostIndicatorsState::getIndexedPages),
                        GrowingNumber.create(diff, HostIndicatorsState::getSearchablePages),
                        GrowingNumber.create(diff, HostIndicatorsState::getExcludedPages),
                        ownerInfo.getLeft(),
                        ownerInfo.getRight()
                );
            }
        }
        ShortHostInfoBuilder builder = new ShortHostInfoBuilder();
        builder.hostname(toCommonHostInfo(hostId, displayName, rtProblems, pinned));
        builder.pinned(pinnedNodes);

        if (verified) {
            if (hostData == null) {
                builder.hostDataStatus(HostDataStatus.HOST_NOT_LOADED).addedToList(true);
            } else {
                builder.hostData(hostData).addedToList(true);
                // TODO каким-то образом в dataState в summary могут оказаться лишние проблемы (типа THREATS)
                // есть надежда, что после от миграции с поколений это все окажется ненужным
                SummarySiteProblemsInfo problemsInfo = rtProblems.merge(dataState.getProblemsSummary().filter(SiteProblemStorageType.GENERATION_BASED));
                builder.problemSeverities(ShortHostInfo.toSeverities(problemsInfo, e -> !e.isTurboProblem()));
                builder.turboProblemSeverities(ShortHostInfo.toSeverities(problemsInfo, SiteProblemTypeEnum.TURBO_PROBLEMS::contains));
                // чудо костыли для WMC-9813
                builder.turboContentProblemSeverities(ShortHostInfo.toSeverities(problemsInfo, problemType ->
                        problemType.isTurboContentProblem(ds.getTurboContentSearchUrls() > 0)));
                builder.turboShopProblemSeverities(ShortHostInfo.toSeverities(problemsInfo, problemType ->
                        problemType.isTurboShopProblem(ds.getTurboShopSearchUrls() > 0)));
            }
        } else {
            builder.hostDataStatus(added ? HostDataStatus.HOST_NOT_VERIFIED : HostDataStatus.HOST_NOT_ADDED).addedToList(added);
        }
        builder.shopTeaser(isInMarket);
        builder.turboInfo(new HostTurboInfo(ds.getTurboSearchUrls(), ds.getTurboUrls(), ds.isHasActiveRssFeeds() || ds.isHasActiveYmlFeeds(), ds.getAutoparserToggleState()));
        builder.turboContentInfo(new HostTurboInfo(ds.getTurboContentSearchUrls(), ds.getTurboUrls(), ds.isHasActiveRssFeeds(), ds.getAutoparserToggleState()));
        builder.turboShopInfo(new HostTurboInfo(ds.getTurboShopSearchUrls(), ds.getTurboUrls(), ds.isHasActiveYmlFeeds(), AutoparserToggleState.ABSENT));
        builder.mirrors(new ArrayList<>());

        builder.metrikaCrawlState(crawlState.getAsMap());
        builder.verticalShare(hasSuccessfulGoodFeeds);

        return builder.build();
    }

    public static CommonHostInfo toCommonHostInfo(WebmasterHostId hostId,
                                                  String displayName,
                                                  SummarySiteProblemsInfo rtProblemsSummary, boolean pinned) {
        return new CommonHostInfo(hostId, displayName, pinned, HostHealth.fromWebmasterProblemSummaries(rtProblemsSummary));
    }
}
