package ru.yandex.webmaster3.viewer.http.user;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import lombok.Builder;
import lombok.Value;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemSeverityEnum;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.turbo.model.autoparser.AutoparserToggleState;
import ru.yandex.webmaster3.storage.checklist.data.SummarySiteProblemsInfo;
import ru.yandex.webmaster3.storage.host.HostIndicatorsDiff;
import ru.yandex.webmaster3.storage.host.HostIndicatorsState;
import ru.yandex.webmaster3.storage.metrika.data.MetrikaCounterCrawlStateEnum;
import ru.yandex.webmaster3.viewer.http.common.response.CommonHostInfo;
import ru.yandex.webmaster3.viewer.http.common.response.HostDataStatus;

/**
 * @author aherman
 */
public abstract class GetHostListResponse implements ActionResponse {

    @Value
    public static class NormalResponse extends GetHostListResponse implements ActionResponse.NormalResponse {
        int verifiedHostCount;
        int unverifiedHostCount;
        int mainMirrorsCount;
        int filteredMainMirrorsCount;

        List<ShortHostInfo> hosts;
        Map<WebmasterHostId, WebmasterHostId> hostsWithoutVerifiedMainMirror;
    }

    @Value
    @Builder
    public static class ShortHostInfo {
        CommonHostInfo hostname;
        List<WebmasterHostId> pinned;
        HostData hostData;
        HostDataStatus hostDataStatus;
        List<ShortHostInfo> mirrors;
        boolean addedToList;
        boolean shopTeaser;
        HostTurboInfo turboInfo;
        HostTurboInfo turboContentInfo;
        HostTurboInfo turboShopInfo;
        @Description("Количество проблем по уровням серьезности")
        Map<SiteProblemSeverityEnum, Integer> problemSeverities;
        Map<SiteProblemSeverityEnum, Integer> turboProblemSeverities;
        Map<SiteProblemSeverityEnum, Integer> turboContentProblemSeverities;
        Map<SiteProblemSeverityEnum, Integer> turboShopProblemSeverities;

        Map<Long, MetrikaCounterCrawlStateEnum> metrikaCrawlState;
        boolean verticalShare;

        public static Map<SiteProblemSeverityEnum, Integer> toSeverities(SummarySiteProblemsInfo summaryInfo,
                                                                         Predicate<SiteProblemTypeEnum> filter) {
            return summaryInfo.getProblem2State().entrySet().stream()
                    .filter(entry -> entry.getValue().isPresent())
                    .filter(entry -> filter.test(entry.getKey()))
                    .map(entry -> entry.getKey().getSeverity())
                    .collect(Collectors.groupingBy(Function.identity(), Collectors.reducing(0, e -> 1, Integer::sum)));
        }

   }

    @Value
    @Builder
    public static class HostData {
        GrowingNumber sqi;
        GrowingNumber indexedPages;
        GrowingNumber searchablePages;
        GrowingNumber excludedPages;
        boolean owner;
        String ownerDomain;
    }

    @Value
    public static class GrowingNumber {
        long value;
        int growsPercent;

        public static GrowingNumber create(HostIndicatorsDiff diff, Function<HostIndicatorsState, Long> mapper) {
            return create(mapper.apply(diff.getPreviousInfo()), mapper.apply(diff.getCurrentInfo()));
        }

        public static GrowingNumber create(long previousValue, long currentValue) {
            if (previousValue == 0) {
                return new GrowingNumber(currentValue, 0);
            }
            float delta = currentValue - previousValue;
            int percent = Math.round(100.0f * (delta) / previousValue);
            return new GrowingNumber(currentValue, percent);
        }
    }

    @Value
    public static class HostTurboInfo {

        @Description("Количество турбо-страниц в поиске")
        long turboSearchPages;
        @Description("Всего турбо-страниц")
        long totalTurboPages;
        @Description("Имеются включенные фиды")
        boolean hasActiveFeeds;
        @Description("Состояние автопарсера - включен/выключен/недоступен")
        AutoparserToggleState autoparserToggleState;
    }
}
