package ru.yandex.webmaster3.viewer.http.user;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.user.UserInitializationInfo;
import ru.yandex.webmaster3.storage.user.service.InitializedUsersService;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.viewer.http.AbstractUserAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserRequest;

import java.util.UUID;

/**
 * @author leonidrom
 */
@Slf4j
@ReadAction
@Category("user")
@Description(value = "Возвращает UUID пользователя по его паспортному id.")
@Component("/user/UUIDById")
public class GetUserUUIDByIdAction extends AbstractUserAction<GetUserUUIDByIdAction.Req, GetUserUUIDByIdAction.Resp> {
    private static final String FAIL_MESSAGE = "Failed to get user initialization info";

    private final InitializedUsersService initializedUsersService;

    @Autowired
    public GetUserUUIDByIdAction(InitializedUsersService initializedUsersService) {
        this.initializedUsersService = initializedUsersService;
    }

    @Override
    public Resp process(Req request) {
        long userId = request.getUserId();
        UserInitializationInfo initInfo;
        try {
            initializedUsersService.initializeUserIfNeeded(userId);
            initInfo = initializedUsersService.getUserInfo(userId);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException(FAIL_MESSAGE,
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }

        if (initInfo == null) {
            throw new WebmasterException(FAIL_MESSAGE,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), FAIL_MESSAGE));
        }

        return new Resp(userId, initInfo.userUuid());
    }



    public static class Req extends AbstractUserRequest {
        @Override
        public boolean isLoggable() {
            return false;
        }
    };

    public static class Resp implements ActionResponse.NormalResponse {
        private final long userId;
        private final UUID userUUID;

        public Resp(long userId, UUID userUUID) {
            this.userId = userId;
            this.userUUID = userUUID;
        }

        @Description("переданный паспортный id пользователя")
        public long getUserId() {
            return userId;
        }

        @Description("UUID пользователя")
        public UUID getUserUUID() {
            return userUUID;
        }
    };
}
