package ru.yandex.webmaster3.viewer.http.user.settings.frontend;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.user.settings.FrontendUserHostSettingsKey;
import ru.yandex.webmaster3.storage.user.settings.FrontendUserHostSettingsYDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

import java.io.IOException;

/**
 * @author leonidrom
 */
@WriteAction
@Category("user")
@Description("Внутрення ручка для удаления подключа из JSON обьекта, хранимого в базе по переданному ключу")
@Component("/user/host/settings/deleteSubkey")
@Slf4j
public class DeleteUserHostSettingsSubkeyAction extends AbstractUserVerifiedHostAction<
        DeleteUserHostSettingsSubkeyAction.DeleteUserHostSettingsSubkeyRequest,
        DeleteUserHostSettingsSubkeyAction.DeleteUserHostSettingsSubkeyResponse> {

    private final static ObjectMapper OM = new ObjectMapper();

    private final FrontendUserHostSettingsYDao frontendUserHostSettingsYDao;

    @Autowired
    public DeleteUserHostSettingsSubkeyAction(FrontendUserHostSettingsYDao frontendUserHostSettingsYDao) {
        this.frontendUserHostSettingsYDao = frontendUserHostSettingsYDao;
    }

    @Override
    public DeleteUserHostSettingsSubkeyResponse process(DeleteUserHostSettingsSubkeyRequest request) {
        var hostId = request.getHostId();
        var userId = request.getUserId();
        var key = request.getKey();

        var keyObjStr = frontendUserHostSettingsYDao.getValue(userId, hostId, key);
        if (StringUtils.isEmpty(keyObjStr)) {
            return new DeleteUserHostSettingsSubkeyResponse();
        }

        try {
            var keyObj = (ObjectNode)OM.readTree(keyObjStr);
            keyObj.remove(request.getSubkey());
            if (keyObj.size() == 0) {
                keyObjStr = null;
            } else {
                keyObjStr = OM.writeValueAsString(keyObj);
            }
        } catch (IOException e) {
            throw new WebmasterException("Failed to parse user settings",
                new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Failed to parse user settings"), e);
        }

        try {
            if (keyObjStr == null) {
                frontendUserHostSettingsYDao.deleteKey(userId, hostId, key);
            } else {
                frontendUserHostSettingsYDao.updateValue(userId, hostId, key, keyObjStr);
            }
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to update user host settings",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }

        return new DeleteUserHostSettingsSubkeyResponse();
    }

    public static class DeleteUserHostSettingsSubkeyRequest extends AbstractUserVerifiedHostRequest {
        private FrontendUserHostSettingsKey key;
        private String subkey;

        public FrontendUserHostSettingsKey getKey() {
            return key;
        }

        @RequestQueryProperty(required = true)
        @Description("Название ключа настроек")
        public void setKey(FrontendUserHostSettingsKey key) {
            this.key = key;
        }

        public String getSubkey() {
            return subkey;
        }

        @RequestQueryProperty(required = true)
        @Description("Подключ который надо удалить")
        public void setSubkey(String subkey) {
            this.subkey = subkey;
        }
    }

    public static class DeleteUserHostSettingsSubkeyResponse implements ActionResponse.NormalResponse {
    }
}
