package ru.yandex.webmaster3.viewer.http.user.settings.notification2;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import lombok.AllArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.notifications.NotificationChannel;
import ru.yandex.webmaster3.storage.notifications.UserHostNotificationConfiguration;
import ru.yandex.webmaster3.storage.notifications.service.UserNotificationSettingsService;
import ru.yandex.webmaster3.storage.user.dao.UserNotificationEmailYDao;
import ru.yandex.webmaster3.storage.user.notification.HostNotificationMode;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;

/**
 * @author aherman
 */
@ReadAction
@Category("notifications")
@AllArgsConstructor(onConstructor_ = @Autowired)
public class GetUserNotification2SettingsPerHostAction extends AbstractUserVerifiedHostAction<GetUserNotification2SettingsPerHostRequest, GetUserNotification2SettingsPerHostResponse> {

    private UserHostsService userHostsService;
    private UserNotificationSettingsService userNotificationSettingsService;
    private UserNotificationEmailYDao userNotificationEmailYDao;

    @Override
    public GetUserNotification2SettingsPerHostResponse process(GetUserNotification2SettingsPerHostRequest request) {
        WebmasterHostId hostId = request.getHostId();
        WebmasterUser user = request.getWebmasterUser();
        boolean pinned = userHostsService.isHostPinned(user, hostId);

        UserHostNotificationConfiguration userNotificationsSettings = userNotificationSettingsService
                .getUserNotificationsSettings(request.getWebmasterUser(), Collections.singletonList(hostId));

        Map<NotificationType, GetUserNotification2SettingsPerHostResponse.HostNotificationTypeSettings> defaultSettings = new HashMap<>();
        for (NotificationType type : UserNotificationSettingsService.PER_HOST_NOTIFICATION_TYPES) {
            String emailMode = null;
            String serviceMode = null;
            String supMode = null;
            if (type.getSupportedChannels().contains(NotificationChannel.EMAIL)) {
                emailMode = userNotificationsSettings.getPerUserSetting(type, NotificationChannel.EMAIL).name();
            }
            if (type.getSupportedChannels().contains(NotificationChannel.SERVICE)) {
                serviceMode = userNotificationsSettings.getPerUserSetting(type, NotificationChannel.SERVICE).name();
            }
            if (type.getSupportedChannels().contains(NotificationChannel.SUP)) {
                supMode = userNotificationsSettings.getPerUserSetting(type, NotificationChannel.SUP).name();
            }
            defaultSettings.put(
                    type,
                    new GetUserNotification2SettingsPerHostResponse.HostNotificationTypeSettings(
                            emailMode,
                            serviceMode,
                            supMode
                    )
            );
        }

        Map<NotificationType, GetUserNotification2SettingsPerHostResponse.HostNotificationTypeSettings> s = new HashMap<>();
        for (NotificationType type : UserNotificationSettingsService.PER_HOST_NOTIFICATION_TYPES) {
            String emailMode = getNotificationStatus(hostId, userNotificationsSettings, type, NotificationChannel.EMAIL);
            String serviceMode = getNotificationStatus(hostId, userNotificationsSettings, type, NotificationChannel.SERVICE);
            String supMode = getNotificationStatus(hostId, userNotificationsSettings, type, NotificationChannel.SUP);
            s.put(
                    type,
                    new GetUserNotification2SettingsPerHostResponse.HostNotificationTypeSettings(
                            emailMode,
                            serviceMode,
                            supMode
                    )
            );
        }
        GetUserNotification2SettingsPerHostResponse.HostSetting settingsResult =
                new GetUserNotification2SettingsPerHostResponse.HostSetting(hostId, pinned, s);
        String email = userNotificationEmailYDao.getEmail(request.getUserId());
        return new GetUserNotification2SettingsPerHostResponse.NormalResponse(defaultSettings, settingsResult, email);
    }

    private String getNotificationStatus(WebmasterHostId hostId, UserHostNotificationConfiguration userNotificationsSettings, NotificationType type, NotificationChannel notificationChannel) {
        String mode = null;
        if (type.getSupportedChannels().contains(notificationChannel)) {
            Optional<HostNotificationMode> modeOpt =
                    userNotificationsSettings.getPerHostSettingOrDefault(hostId, type, notificationChannel);
            if (modeOpt.isEmpty()) {
                throw new WebmasterException("Unknown host " + hostId,
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(),
                                "Unknown host " + hostId));
            }
            mode = modeOpt.get().name();
        }
        return mode;
    }
}
