package ru.yandex.webmaster3.viewer.microdata;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.L10nEnum;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.viewer.microdata.data.MicrodataByDocumentResult;
import ru.yandex.webmaster3.viewer.microdata.data.MicrodataByUrlResult;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;

/**
 * @author avhaliullin
 */
public class MicrodataService {
    private static final Logger log = LoggerFactory.getLogger(MicrodataService.class);

    private static final Charset UTF8 = Charset.forName("UTF-8");
    private static final ObjectMapper OM = new ObjectMapper();
    private static final byte[] CONTENT_PARAM = "doctext=".getBytes(UTF8);

    private static final String COMPLEX_DOCUMENT = "COMPLEX_DOCUMENT";
    private static final String MISSING_LANGUAGE = "MISSING_LANGUAGE";
    private static final String UNSUPPORTED_LANGUAGE = "UNSUPPORTED_LANGUAGE";
    private static final String DOCUMENT_REQUEST_IS_INVALID = "DOCUMENT_REQUEST_IS_INVALID";
    private static final String MISSING_URL = "MISSING_URL";
    private static final String INVALID_URL = "INVALID_URL";
    private static final String URL_DOWNLOADING_ERROR = "URL_DOWNLOADING_ERROR";

    // TODO подумать о таймаутах, чтобы viewer не зависал
    private final CloseableHttpClient client = HttpClientBuilder.create().build();

    private String microdataUrl;

    public MicrodataByUrlResult validateByUrl(URL url, L10nEnum l10n, String reqId) {
        String requestString = microdataUrl + "parseUrlv2?" + "lang=" + l10n.getName() + "&url=" + urlencode(url.toExternalForm()) + "&id=" + urlencode(reqId);
        log.info("Querying validator with {}", requestString);
        HttpGet req = new HttpGet(requestString);
        try (CloseableHttpResponse res = client.execute(req)) {
            InputStream is = res.getEntity().getContent();
            JsonNode root = OM.readTree(is);
            switch (res.getStatusLine().getStatusCode()) {
                case 200:
                    if (!(root instanceof ObjectNode)) {
                        throw makeError("Microdata validator returned non-object json with 200 code ", reqId, root);
                    } else {
                        return new MicrodataByUrlResult(MicrodataByUrlResult.Status.OK, (ObjectNode) root);
                    }
                case 400:
                case 504:
                    if (!root.has("errorCode")) {
                        throw makeError("validator returned 400 without errorCode", reqId, root);
                    }
                    String errorCode = root.get("errorCode").textValue();
                    switch (errorCode) {
                        case COMPLEX_DOCUMENT:
                            return new MicrodataByUrlResult(MicrodataByUrlResult.Status.COMPLEX_DOCUMENT, null);
                        case URL_DOWNLOADING_ERROR:
                            return new MicrodataByUrlResult(MicrodataByUrlResult.Status.FAILED_TO_DOWNLOAD, null);
                        case UNSUPPORTED_LANGUAGE:
                            throw makeError("Validator reported unsupported language. Language = " + l10n, reqId, root);
                        default:
                            throw makeError("Unexpected validator error " + errorCode, reqId, root);
                    }
                default:
                    throw makeError("Unknown validator response code: " + res.getStatusLine().getStatusCode(), reqId, root);
            }
        } catch (IOException e) {
            throw new WebmasterException("Failed to validate document by url; reqId=" + reqId,
                    new WebmasterErrorResponse.MicrodataServiceErrorResponse(MicrodataService.class), e);
        }
    }

    private WebmasterException makeError(String message, String reqId, JsonNode json) {
        return new WebmasterException(message + "; reqId=" + reqId + "\n" + json,
                new WebmasterErrorResponse.MicrodataServiceErrorResponse(MicrodataService.class));
    }

    public MicrodataByDocumentResult validateByDocument(String documentContent, L10nEnum l10n, String reqId) {
        String requestString = microdataUrl + "parseHtmlv2?" + "lang=" + l10n.getName() + "&id=" + urlencode(reqId);
        log.info("Querying validator with {}", requestString);
        HttpPost req = new HttpPost(requestString);
        String encodedDocument = urlencode(documentContent);
        InputStream entityIS = new SequenceInputStream(new ByteArrayInputStream(CONTENT_PARAM), new ByteArrayInputStream(encodedDocument.getBytes(UTF8)));
        HttpEntity entity = new InputStreamEntity(entityIS, ContentType.APPLICATION_FORM_URLENCODED);
        req.setEntity(entity);
        try (CloseableHttpResponse res = client.execute(req)) {
            InputStream is = res.getEntity().getContent();
            JsonNode root = OM.readTree(is);
            switch (res.getStatusLine().getStatusCode()) {
                case 200:
                    if (!(root instanceof ObjectNode)) {
                        throw makeError("Microdata validator returned non-object json with 200 code ", reqId, root);
                    } else {
                        return new MicrodataByDocumentResult(MicrodataByDocumentResult.Status.OK, (ObjectNode) root);
                    }
                case 400:
                    if (!root.has("errorCode")) {
                        throw makeError("validator returned 400 without errorCode", reqId, root);
                    }
                    String errorCode = root.get("errorCode").textValue();
                    switch (errorCode) {
                        case COMPLEX_DOCUMENT:
                            return new MicrodataByDocumentResult(MicrodataByDocumentResult.Status.COMPLEX_DOCUMENT, null);
                        case UNSUPPORTED_LANGUAGE:
                            throw makeError("Validator reported unsupported language. Language = " + l10n, reqId, root);
                        default:
                            throw makeError("Unexpected validator error " + errorCode, reqId, root);
                    }
                default:
                    throw makeError("Unknown validator response code: " + res.getStatusLine().getStatusCode(), reqId, root);
            }
        } catch (IOException e) {
            throw new WebmasterException("Failed to validate document by url",
                    new WebmasterErrorResponse.MicrodataServiceErrorResponse(MicrodataService.class), e);
        }
    }

    private String urlencode(String s) {
        try {
            return URLEncoder.encode(s, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Should never happer", e);
        }
    }

    @Required
    public void setMicrodataUrl(String microdataUrl) {
        this.microdataUrl = microdataUrl;
    }
}
