package ru.yandex.webmaster3.viewer.microdata.semantic;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.SemanticAPI_v1;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.microformats.FrontEnd;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.rdfa.exceptions.RDFaDisallowKeyException;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.serialize.util.APIVersion;
import ru.yandex.webmaster3.core.semantic.semantic_document_parser.tanker.LanguageContext;
import ru.yandex.webmaster3.core.zora.ZoraForValidatorsService;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;
import ru.yandex.webmaster3.viewer.microdata.MicrodataService;


@Component
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Slf4j
public class SemanticValidation {
    private static final ObjectMapper OM = JsonDBMapping.OM;
    private final ZoraForValidatorsService zoraForValidatorsService;

    private final FrontEnd frontEnd;

    public Map<String, JsonNode> validateByDocument(String document, String lang, String id, boolean errorsOnly) {
        return validate(document, "http://localhost", lang, id, errorsOnly);
    }

    public Map<String, JsonNode> validateByUrl(URL url, String lang, String reqId, boolean errorsOnly) {
        String content = zoraForValidatorsService.downloadDocumentAsStringGoZora(url.toString());
        return validate(content, url.toString(), lang, reqId, errorsOnly);
    }

    private Map<String, JsonNode> validate(String content, String url, String lang, String reqId, boolean errorsOnly) {
        try {
            Map<String, List<JSONObject>> original = new SemanticAPI_v1().publicAPIResult(content,
                    url, frontEnd, getLanguageContext(lang), reqId, errorsOnly, APIVersion.VERSION_1_1).
                    convertToMap(APIVersion.VERSION_1_1);
            Map<String, JsonNode> result = new HashMap<>(original.size());
            for (Map.Entry<String, List<JSONObject>> item : original.entrySet()) {
                result.put(item.getKey(), OM.readTree(item.getValue().toString()));
            }
            return result;
        } catch (JSONException | RDFaDisallowKeyException | IOException e) {
            throw new WebmasterException("Failed to validate document by url; reqId=" + reqId,
                    new WebmasterErrorResponse.MicrodataServiceErrorResponse(MicrodataService.class), e);
        }
    }

    private LanguageContext getLanguageContext(final String lang) {
        if (frontEnd.getContextMap().containsKey(lang)) {
            return frontEnd.getContextMap().get(lang);
        }
        return null;
    }
}
