package ru.yandex.webmaster3.viewer.video;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.http.*;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.zora.GoZoraService;
import ru.yandex.webmaster3.core.zora.go_data.request.GoZoraRequest;
import ru.yandex.webmaster3.storage.video.VideohostOfferService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;
import ru.yandex.wmtools.common.util.uri.WebmasterUriUtils;

import javax.annotation.Nullable;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * @author vsedaikina
 * 18.10.21
 */
@Slf4j
@Component("/video/updateVideohostOffer")
@WriteAction
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Category("video")
public class UpdateVideohostOfferAction extends AbstractUserVerifiedHostAction<UpdateVideohostOfferAction.Req, UpdateVideohostOfferAction.Res> {
    private final VideohostOfferService videohostOfferService;
    private final GoZoraService goZoraService;

    @Override
    public Res process(Req request) {
        String url = canonizeUrl(request.videoOfferUrl);
        if (url == null) {
            log.info("Invalid offer url");
            return new Res.InvalidUrlResponse();
        }

        if (!IdUtils.toDomainHostId(IdUtils.urlToHostId(url)).equals(IdUtils.toDomainHostId(request.getHostId()))) {
            log.info("Host offer and domain mismatch");
            return new Res.HostAndOfferDomainMismatchResponse();
        }

        Res res = checkUrl(url);
        if (res != null) {
            return res;
        }

        videohostOfferService.updateVideohostOfferUrl(request.getHostId(), request.videoOfferUrl, request.isRechecking);

        return new Res.NormalResponse();
    }

    private Res checkUrl(String url) {
        var zoraResponse = goZoraService.executeRequestFetchResponse(new GoZoraRequest(url, false, true, false));

        if (zoraResponse.getHttpCode() / 100 != 2) {
            log.info("Bad url http code response {}", zoraResponse.getHttpCode());
            return new Res.BadUrlHttpCodeResponse();
        }

        return null;
    }

    @Nullable
    public static String canonizeUrl(String requestUrl) {
        URL url;
        try {
            url = new URL(requestUrl);
        } catch (MalformedURLException e) {
            return null;
        }
        return url.toString();
    }


    public static final class Req extends AbstractUserVerifiedHostRequest implements ActionRequest {
        @Setter(onMethod_ = @RequestQueryProperty())
        boolean isRechecking;
        @Setter(onMethod_ = @RequestQueryProperty(required = true))
        String videoOfferUrl;
    }

    public abstract static class Res implements ActionResponse {

        public static class NormalResponse extends Res implements ActionResponse.NormalResponse {
        }

        public enum ErrorCode {
            INVALID_URL,
            BAD_URL_HTTP_CODE,
            HOST_AND_OFFER_DOMAIN_MISMATCH
        }

        public abstract static class AbstractAddOfferError extends Res implements ActionResponse.ErrorResponse {
            private final ErrorCode code;

            public AbstractAddOfferError(ErrorCode code) {
                this.code = code;
            }

            @Override
            public Enum<?> getCode() {
                return code;
            }
        }

        @Description("Урл в неверном формате")
        public static class InvalidUrlResponse extends Res.AbstractAddOfferError {
            public InvalidUrlResponse() {
                super(ErrorCode.INVALID_URL);
            }
        }

        @Description("Попробовали скачать фид но у него код != 2хх")
        public static class BadUrlHttpCodeResponse extends Res.AbstractAddOfferError {
            public BadUrlHttpCodeResponse() {
                super(ErrorCode.BAD_URL_HTTP_CODE);
            }
        }

        @Description("Домены хоста и фоерты не совападают")
        public static class HostAndOfferDomainMismatchResponse extends Res.AbstractAddOfferError {
            public HostAndOfferDomainMismatchResponse() {
                super(ErrorCode.HOST_AND_OFFER_DOMAIN_MISMATCH);
            }
        }
    }
}
