package ru.yandex.webmaster3.worker;

import java.util.UUID;

import com.google.common.base.Stopwatch;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.tracer.BeautyYdbTrace;
import ru.yandex.webmaster3.core.tracer.YdbMetricsService;
import ru.yandex.webmaster3.core.tracer.YdbTracer;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.logging.ExceptionWrapper;
import ru.yandex.webmaster3.storage.logging.TasksLoggingService;
import ru.yandex.webmaster3.worker.task.periodic.PeriodicTaskMetrics;

/**
 * @author tsyplyaev
 */

public abstract class PeriodicTask<TS extends PeriodicTaskState> {
    private static final Logger log = LoggerFactory.getLogger(PeriodicTask.class);

    private volatile DateTime lastStarted;
    private volatile DateTime lastFinished;
    private volatile long lastRunTimeMs;
    private volatile TaskResult lastResult;

    private volatile boolean running = false;
    protected volatile TS state = null;


    @Autowired
    private TasksLoggingService periodicTasksLoggingService;
    @Autowired
    private PeriodicTaskMetrics periodicTaskMetrics;

    public static class Result {
        public final static Result SUCCESS = new Result(TaskResult.SUCCESS);
        public final static Result FAIL = new Result(TaskResult.FAIL);

        private final TaskResult taskResult;

        public Result(TaskResult taskResult) {
            this.taskResult = taskResult;
        }

        public TaskResult getTaskResult() {
            return taskResult;
        }
    }

    public abstract Result run(UUID runId) throws Exception;

    public abstract PeriodicTaskType getType();

    public Result execute(UUID runId) throws Exception {
        running = true;

        boolean disableLogging = getClass().isAnnotationPresent(PeriodicTasksLoggingDisable.class);

        Stopwatch runTime = Stopwatch.createStarted();
        long startTime = System.nanoTime();
        Exception exception = null;
        TaskResult taskResult = TaskResult.UNKNOWN;
        YdbTracer.startTrace();
        try {
            lastStarted = DateTime.now();
            if (!disableLogging) {
                try {
                    periodicTasksLoggingService.logTaskStart(lastStarted, getType().name(), runId);
                } catch (WebmasterException ex) {
                    log.error("Could not log task start", ex);
                }
            }

            Result res = run(runId);
            taskResult = res.getTaskResult();
            lastResult = taskResult;

            return res;
        } catch (Exception ex) {
            exception = ex;
            taskResult = TaskResult.FAIL;
            lastResult = taskResult;

            throw ex;
        } finally {
            lastFinished = DateTime.now();
            lastRunTimeMs = lastFinished.getMillis() - lastStarted.getMillis();
            periodicTaskMetrics.taskFinish(getType(), taskResult, System.nanoTime() - startTime);

            var ydbTrace = YdbTracer.stopTrace();
            if (!disableLogging) {
                try {
                    YdbMetricsService.saveSolomon(ydbTrace);

                    BeautyYdbTrace beautyYdbTrace = ydbTrace != null ? new BeautyYdbTrace(ydbTrace) : null;
                    log.info("Request YDB trace pretty: {} {}", getType(), beautyYdbTrace);
                    log.info("Request YDB trace json: {} {}", getType(), JsonMapping.writeValueAsString(ydbTrace));

                    periodicTasksLoggingService
                            .logTaskFinish(lastFinished, getType().name(), taskResult, runId, lastRunTimeMs, state,
                                    wrapException(exception), beautyYdbTrace);
                } catch (WebmasterException ex) {
                    log.error("Could not log task finish", ex);
                }
            }
            running = false;
        }
    }

    private Object wrapException(Exception exception) {
        return exception != null ? new ExceptionWrapper(exception) : null;
    }

    public TaskLockMode getLockMode() {
        return TaskLockMode.ONE_PER_CLUSTER;
    }

    public abstract TaskSchedule getSchedule();

    public TS getState() {
        return state;
    }

    protected void setState(TS state) {
        this.state = state;
    }

    public boolean isRunning() {
        return running;
    }

    public DateTime getLastStarted() {
        return lastStarted;
    }

    public DateTime getLastFinished() {
        return lastFinished;
    }

    public long getLastRunTimeMs() {
        return lastRunTimeMs;
    }

    public TaskResult getLastResult() {
        return lastResult;
    }

    public void setPeriodicTasksLoggingService(TasksLoggingService periodicTasksLoggingService) {
        this.periodicTasksLoggingService = periodicTasksLoggingService;
    }

    public void setPeriodicTaskMetirics(PeriodicTaskMetrics periodicTaskMetrics) {
        this.periodicTaskMetrics = periodicTaskMetrics;
    }

}
