package ru.yandex.webmaster3.worker.abt;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.Range;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskData;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskType;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.abt.ExperimentMapperService;
import ru.yandex.webmaster3.storage.abt.dao.AbtHostExperimentYDao;
import ru.yandex.webmaster3.storage.abt.dao.AbtUserExperimentYDao;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.abt.model.ExperimentInfo;
import ru.yandex.webmaster3.storage.abt.model.ExperimentScope;
import ru.yandex.webmaster3.storage.abt.model.IExperiment;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.worker.Task;

/**
 * @author akhazhoyan 06/2018
 * <p>
 * Импортирует данные про А/Б тесты из YT в Кассандру.
 * Данные в YT лежат в Банахе в {@code experiments/$SCOPE/$EXPERIMENT},
 * где SCOPE равен имени одного из {@link ExperimentScope},
 * а EXPERIMENT равен имени одного из {@link Experiment} (в нижнем регистре).
 * <p>
 * Таска принимает на вход {@link Experiment}, что однозначно определяет таблицу в YT.
 * Данные закачиваются в Кассандру, таблица выбирается в зависимости от {@link ExperimentScope}.
 */
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Component("importExperimentsTask")
public final class ImportExperimentsTask extends Task<ImportExperimentsTask.Data> {
    public static final String PATH_DELIMITER = "/";

    @Value("${external.yt.service.arnold.root.default}/experiments")
    private String rootPath;
    private final YtService ytService;
    private final AbtUserExperimentYDao abtUserExperimentYDao;
    private final AbtHostExperimentYDao abtHostExperimentYDao;
    private final ExperimentMapperService experimentMapperService;

    private YtPath tablePath(IExperiment experiment) {
        String path = experiment.getScope().name().toLowerCase() + PATH_DELIMITER + experiment.getName().toLowerCase();
        return YtPath.path(YtPath.fromString(rootPath), path);
    }

    @Override
    public Result run(Data data) {
        final IExperiment experiment = experimentMapperService.getExperiment(data.experiment);
        YtPath tablePath = tablePath(experiment);
        ytService.inTransaction(tablePath).execute(ytCypressService -> {
            switch (experiment.getScope()) {
                case DOMAIN:
                case HOST:
                    return process(ytCypressService, experiment, YtHostExperimentRow.class, this::processYtHostExperimentBatchRows);
                case USER:
                    return process(ytCypressService, experiment, YtUserExperimentRow.class, this::processYtUserExperimentBatchRows);
                default:
                    throw new IllegalStateException("Unexpected enum value " + experiment.getScope());
            }
        });
        return new Result(TaskResult.SUCCESS);
    }

    private interface RowProcessor<T> {
        void process(IExperiment data, T row);
    }

    private <T> boolean process(
            YtCypressService ytCypressService,
            IExperiment experiment,
            Class<T> cls,
            RowProcessor<List<T>> rowProcessor) {
        AsyncTableReader<T> tableReader = new AsyncTableReader<>(
                ytCypressService,
                tablePath(experiment),
                Range.all(),
                YtTableReadDriver.createYSONDriver(cls)
        ).splitInParts(10000L).withRetry(5);
        List<T> rowsForProcess = new ArrayList<>();
        try (AsyncTableReader.TableIterator<T> iter = tableReader.read()) {
            while (iter.hasNext()) {
                rowsForProcess.add(iter.next());
                if (rowsForProcess.size() > 4096) {
                    rowProcessor.process(experiment, rowsForProcess);
                    rowsForProcess.clear();
                }
            }
            if (rowsForProcess.size() > 0) {
                rowProcessor.process(experiment, rowsForProcess);
            }
            return true;
        } catch (InterruptedException | IOException e) {
            Thread.currentThread().interrupt();
            throw new YtException(e);
        } catch (WebmasterYdbException e) {
            throw e.asUnchecked("Failed to load experiment data into Cassandra", getClass());
        }
    }

    private void processYtUserExperimentBatchRows(IExperiment experiment, List<YtUserExperimentRow> rows) {
        final List<Pair<Long, ExperimentInfo>> items = rows.stream().map(row -> Pair.of(row.userId, new ExperimentInfo(experiment.getName(), row.group))).collect(Collectors.toList());
        abtUserExperimentYDao.batchInsert(items);
    }

    private void processYtHostExperimentBatchRows(IExperiment experiment, List<YtHostExperimentRow> rows) {
        final List<Pair<WebmasterHostId, ExperimentInfo>> items = rows.stream()
                .map(row -> Pair.of(row.hostId, new ExperimentInfo(experiment.getName(), row.group)))
                .collect(Collectors.toList());
        abtHostExperimentYDao.batchInsert(items);
    }

    @Override
    public Class<Data> getDataClass() {
        return Data.class;
    }


    private static class YtUserExperimentRow {
        final long userId;
        final String group;

        @JsonCreator
        public YtUserExperimentRow(@JsonProperty("user_id") long userId,
                                   @JsonProperty("group") String group) {
            this.userId = userId;
            this.group = group;
        }
    }

    private static class YtHostExperimentRow {
        final WebmasterHostId hostId;
        final String group;

        @JsonCreator
        public YtHostExperimentRow(@JsonProperty("host_id") String hostIdStr,
                                   @JsonProperty("group") String group) {
            this.hostId = IdUtils.urlToHostId(hostIdStr);
            this.group = group;
        }
    }

    public static class Data extends WorkerTaskData {
        final String experiment;

        public Data(@JsonProperty("experiment") String experiment) {
            this.experiment = experiment;
        }

        @Override
        public WorkerTaskType getTaskType() {
            return WorkerTaskType.IMPORT_EXPERIMENTS;
        }

        @Override
        public String getShortDescription() {
            return "Importing A/B testing data from YT to Cassandra";
        }
    }
}
