package ru.yandex.webmaster3.worker.addurl;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.commons.lang3.mutable.MutableObject;
import org.joda.time.DateTime;
import org.joda.time.Days;
import org.joda.time.Duration;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.addurl.UrlRecrawlEventLog;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.addurl.AddUrlEventsLogsYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.util.yt.*;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Created by ifilippov5 on 31.10.17.
 */
public class UploadAddUrlLogsPeriodicTask extends PeriodicTask<UploadAddUrlLogsPeriodicTask.TaskState> {
    private static final Logger log = LoggerFactory.getLogger(UploadAddUrlLogsPeriodicTask.class);

    private static final Duration MAX_TABLE_AGE = Duration.standardDays(3);
    private static final String YT_TABLE_PREFIX = "addurl_events_log";
    private static final Pattern RESULT_TABLE_NAME_PATTERN = Pattern.compile(YT_TABLE_PREFIX + "\\.(\\d+)");
    private static final DateTimeFormatter DATE_FORMAT_ONLY_DAY = DateTimeFormat.forPattern("yyyyMMdd");

    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(10, Duration.standardMinutes(2));

    private static final String TABLE_SCHEMA = "[" +
            "{'name': '" + YtRow.F_URL + "', 'type': 'string'}, " +
            "{'name': '" + YtRow.F_USER_IP + "', 'type': 'string'}, " +
            "{'name': '" + YtRow.F_USER_ID + "', 'type': 'uint64'}, " +
            "{'name': '" + YtRow.F_STATE + "', 'type': 'string'}]";

    private AddUrlEventsLogsYDao addUrlEventsLogsYDao;
    private YtPath workDir;
    private YtService ytService;

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new UploadAddUrlLogsPeriodicTask.TaskState());

        MutableObject<List<DateTime>> datesToProcess = new MutableObject<>();
        try {
            ytService.withoutTransaction(cypressService -> {
                datesToProcess.setValue(findDatesToProcess(cypressService.list(workDir), DateTime.now()));
                return true;
            });
        } catch (YtException e) {
            throw new WebmasterException("YT error",
                    new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
        }

        if (datesToProcess.getValue() != null && !datesToProcess.getValue().isEmpty()) {
            fillTables(datesToProcess.getValue());
        }

        return new Result(TaskResult.SUCCESS);
    }

    private void fillTables(List<DateTime> datesToProcess) throws Exception {
        Map<DateTime, List<UrlRecrawlEventLog>> logs = new HashMap<>();
        for (DateTime date : datesToProcess) {
            logs.put(date, new ArrayList<>());
        }

        log.info("Start download from YDB");
        MutableInt eventsCount = new MutableInt(0);
        try {
            addUrlEventsLogsYDao.foreachEvent(event -> {
                DateTime eventDate = event.getUpdateDate().withTimeAtStartOfDay();
                if (logs.containsKey(eventDate)) {
                    logs.get(eventDate).add(event);
                    eventsCount.increment();
                }
            });
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Unable to get from YDB",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
        log.info("{} events downloaded from YDB", eventsCount.getValue());

        Stopwatch stopwatch = Stopwatch.createStarted();
        for (Map.Entry<DateTime, List<UrlRecrawlEventLog>> entry : logs.entrySet()) {
            DateTime date = entry.getKey();
            List<UrlRecrawlEventLog> eventLogs = entry.getValue();
            if (eventLogs.isEmpty()) {
                log.info("For date {} log data is empty", date.toString(DATE_FORMAT_ONLY_DAY));
                continue;
            }
            String ytTableSuffix = date.toString(DATE_FORMAT_ONLY_DAY);
            String tableName = YT_TABLE_PREFIX + "." + ytTableSuffix;
            log.info("Start upload to YT table: {}", tableName);
            YtPath tablePath = YtPath.path(workDir, tableName);

            YtTableData table = null;
            try {
                table = ytService.prepareTableData(tablePath.getName(), tw -> writeToTable(tw, eventLogs));

                YtTransactionService.TransactionProcess process = new YtUtils.TransactionWriterBuilder(tablePath, table)
                        .withSchema(TABLE_SCHEMA)
                        .withRetry(RETRY_POLICY)
                        .build();

                YtUtils.TransactionExecutor writer = new YtUtils.TransactionExecutor(ytService, workDir);
                writer.execute(process);

            } finally {
                if (table != null) {
                    table.delete();
                }
            }
            log.info("Uploaded {} rows", eventLogs.size());
        }
        getState().uploadToYtMs = stopwatch.elapsed(TimeUnit.MILLISECONDS);
    }

    private void writeToTable(TableWriter tw, List<UrlRecrawlEventLog> logs) {
        for (UrlRecrawlEventLog log : logs) {
            tw.column(YtRow.F_URL, log.getFullUrl());
            tw.column(YtRow.F_USER_IP, log.getUserIp());
            tw.column(YtRow.F_USER_ID, log.getUserId());
            tw.column(YtRow.F_STATE, log.getStatus().name());
            try {
                tw.rowEnd();
            } catch (YtException e) {
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPLOAD_ADD_URL_EVENT_LOGS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 * * * *");
    }

    @Required
    public void setAddUrlEventsLogsYDao(AddUrlEventsLogsYDao addUrlEventsLogsYDao) {
        this.addUrlEventsLogsYDao = addUrlEventsLogsYDao;
    }

    @Required
    public void setWorkDir(YtPath workDir) {
        this.workDir = workDir;
    }

    @Required
    public void setYtService(YtService ytService) {
        this.ytService = ytService;
    }

    public static List<DateTime> findDatesToProcess(List<YtPath> existsTables, DateTime now) {
        now = now.withTimeAtStartOfDay();
        long uploadPeriodDurationInDays = MAX_TABLE_AGE.getStandardDays();

        Optional<YtPath> lastTableOpt = existsTables.stream()
                .filter(table -> RESULT_TABLE_NAME_PATTERN.matcher(table.getName()).matches())
                .max(Comparator.naturalOrder());

        if (lastTableOpt.isEmpty()) {
            log.info("Last table is not present");
        } else {
            log.info("Last table: {}", lastTableOpt.get());
            YtPath lastTable = lastTableOpt.get();
            Matcher matcher = RESULT_TABLE_NAME_PATTERN.matcher(lastTable.getName());
            Preconditions.checkState(matcher.matches());
            DateTime lastTableDate = DATE_FORMAT_ONLY_DAY.parseDateTime(matcher.group(1));
            long notUploadedDaysCount =
                    Math.max(Days.daysBetween(lastTableDate, now).getDays() - 1, 0);// логи за сегодняшний день нам не нужны
            uploadPeriodDurationInDays = Math.min(uploadPeriodDurationInDays, notUploadedDaysCount);
        }
        return Stream.iterate(now, date -> date.minus(Duration.standardDays(1)))
                .skip(1) //сегодняшний день пропускаем
                .limit(uploadPeriodDurationInDays)
                .collect(Collectors.toList());
    }

    private static class YtRow {
        static final String F_URL = "url";
        static final String F_USER_IP = "user_ip";
        static final String F_USER_ID = "user_id";
        static final String F_STATE = "state";
    }

    public static class TaskState implements PeriodicTaskState {
        long uploadToYtMs;

        public long getUploadToYtMs() {
            return uploadToYtMs;
        }
    }
}
