package ru.yandex.webmaster3.worker.addurl;

import lombok.Data;
import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.addurl.AddUrlService;
import ru.yandex.webmaster3.core.addurl.RecrawlState;
import ru.yandex.webmaster3.core.addurl.UrlForRecrawl;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.addurl.AddUrlRequestsService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.*;

/**
 * @author leonidrom
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component
public class UrlRecrawlResendStaleRequestsPeriodicTask extends PeriodicTask<UrlRecrawlResendStaleRequestsPeriodicTask.TaskState> {
    private static final Duration RESEND_AFTER = Duration.standardHours(5);
    public static final int STOP_RESENDING_TRESHOLD = 10_000;

    private final AddUrlRequestsService addUrlRequestsService;
    private final AddUrlService addUrlService;

    @Override
    public Result run(UUID runId) throws Exception {
        TaskState taskState = new TaskState();
        setState(taskState);

        // получим все запросы, которые надо перезаслать
        DateTime now = DateTime.now();
        DateTime staleBefore = now.minus(UrlForRecrawl.STALE_REQUEST_AGE);
        DateTime resendBefore = now.minus(RESEND_AFTER);
        var toResend  = addUrlRequestsService.listUnprocessed(staleBefore, resendBefore).stream()
                .filter(req -> (!req.getState().isTerminal() && req.getProcessedDate().isBefore(resendBefore)))
                .map(req -> req.changeState(RecrawlState.IN_PROGRESS, now))
                .limit(STOP_RESENDING_TRESHOLD)
                .toList();

        // высока вероятность, что в контуре поломка, не будем добивать его ретраями
//        if (toResend.size() > STOP_RESENDING_TRESHOLD) {
//            return Result.SUCCESS;
//        }

        // перезашлем запросы
        toResend.forEach(r -> addUrlService.requestRecrawl(r.getFullUrl()));
        addUrlRequestsService.updateBatch(toResend);

        taskState.urlsSentToRecrawl = toResend.size();

        return Result.SUCCESS;
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.ADDURL_RESEND_STALE_REQUESTS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 15 0/2 * * *");
    }

    @Data
    public static class TaskState implements PeriodicTaskState {
        public int urlsSentToRecrawl;
    }
}