package ru.yandex.webmaster3.worker.addurl;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.addurl.AddUrlService;
import ru.yandex.webmaster3.core.addurl.RecrawlState;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.worker.task.UrlRecrawlTaskData;
import ru.yandex.webmaster3.storage.addurl.AddUrlRequestsService;
import ru.yandex.webmaster3.storage.spam.ISpamHostFilter;
import ru.yandex.webmaster3.worker.Task;

/**
 * @author leonidrom
 */
@Category("addurl")
@Component
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UrlRecrawlTask extends Task<UrlRecrawlTaskData> {
    private final ISpamHostFilter fastSpamHostFilter;
    private final AddUrlRequestsService addUrlRequestsService;
    private final AddUrlService addUrlService;

    @Override
    public Result run(UrlRecrawlTaskData data) throws Exception {
        var urlForRecrawl = data.getUrlForRecrawl();
        var req = addUrlRequestsService.get(urlForRecrawl.getHostId(), urlForRecrawl.getUrlId());
        if (req == null) {
            log.error("No request found for {}", urlForRecrawl);
            return Result.FAIL;
        }

        if (req.getState() != RecrawlState.NEW) {
            log.info("Skipping duplicate request: {}", urlForRecrawl);
            return Result.SUCCESS;
        }

        if (fastSpamHostFilter.checkHost(data.getHostId())) {
            log.info("Ignoring spam host {}", data.getHostId());
            // пометим сразу IN_PROGRESS, дабы не мешать мониторингу
            addUrlRequestsService.update(urlForRecrawl.changeState(RecrawlState.IN_PROGRESS, DateTime.now()));
            return Result.SUCCESS;
        }

        addUrlService.requestRecrawl(urlForRecrawl.getFullUrl());
        addUrlRequestsService.update(urlForRecrawl.changeState(RecrawlState.IN_PROGRESS, DateTime.now()));

        return Result.SUCCESS;
    }

    @Override
    public Class<UrlRecrawlTaskData> getDataClass() {
        return UrlRecrawlTaskData.class;
    }
}
