package ru.yandex.webmaster3.worker.antispam;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.collect.Range;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent.SslCertificateError;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.AllHostsCacheService;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;


/**
 * Created by Oleg Bazdyrev on 24/01/2019.
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class ImportSslCertificateErrorsTask extends PeriodicTask<ImportSslCertificateErrorsTask.TaskState> {
    private static final String SSL_CERTIFICATE_THREAT_NAME = "technology.ssl.broken";
    private static final int BATCH_SIZE = 2_000;
    private static final int TOTAL_THREADS = 8;


    private final SiteProblemsService siteProblemsService;
    private final AllHostsCacheService allHostsCacheService;
    private final YtService ytService;

    @Value("${webmaster3.worker.antispam.import.tablePath}")
    private YtPath tablePath;

    @Override
    public Result run(UUID runId) throws Exception {
        TaskState ts = new TaskState();
        setState(ts);

        DateTime dateTime = DateTime.now();
        ytService.withoutTransaction(cypressService -> {
            // просто читаем, фильтруем и генерируем проблемы
            AsyncTableReader<YtRow> tableReader = new AsyncTableReader<>(cypressService, tablePath, Range.all(),
                    YtTableReadDriver.createYSONDriver(YtRow.class))
                    .splitInParts(200_000).withRetry(3);

            try (AsyncTableReader.TableIterator<YtRow> iterator = tableReader.read()) {
                Map<WebmasterHostId, ProblemSignal> batch = new HashMap<>(BATCH_SIZE);
                while (iterator.hasNext()) {
                    YtRow row = iterator.next();
                    if (!SSL_CERTIFICATE_THREAT_NAME.equals(row.getThreat())) {
                        continue;
                    }
                    processRow(row, batch);
                    if (batch.size() > BATCH_SIZE) {
                        state.totalHosts += batch.size();
                        siteProblemsService.updateCleanableProblems(batch, SiteProblemTypeEnum.SSL_CERTIFICATE_ERROR);
                        batch.clear();
                    }
                }
                siteProblemsService.updateCleanableProblems(batch, SiteProblemTypeEnum.SSL_CERTIFICATE_ERROR);
            } catch (IOException e) {
                throw new WebmasterException("Yt IO error",
                        new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
            }

            return true;
        });

        siteProblemsService.notifyCleanableProblemUpdateFinished(SiteProblemTypeEnum.SSL_CERTIFICATE_ERROR, dateTime);

        return new Result(TaskResult.SUCCESS);
    }

    private void processRow(YtRow row, Map<WebmasterHostId, ProblemSignal> map) {
        String domain = row.getHost();
        // хост естественно только https (порт?)
        List<WebmasterHostId> problemHosts = IdUtils.allHttpsHostsForDomain(domain)
                .stream()
                .filter(h -> allHostsCacheService.contains(h))
                .collect(Collectors.toList());

        ProblemSignal problem = new ProblemSignal(new SslCertificateError(), DateTime.now());
        for (WebmasterHostId problemHost : problemHosts) {
            map.put(problemHost, problem);
        }
    }


    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.IMPORT_SSL_CERTIFICATE_ERRORS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 27 3/6 * * *");
    }

    @Required
    public void setTablePath(YtPath tablePath) {
        this.tablePath = tablePath;
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class YtRow {

        private final String host;
        private final String threat;

        @JsonCreator
        public YtRow(String host, String threat) {
            this.host = host;
            this.threat = threat;
        }

        public String getHost() {
            return host;
        }

        public String getThreat() {
            return threat;
        }
    }

    @Getter
    public class TaskState implements PeriodicTaskState {
        int totalHosts;
        int totalFailedHosts;
    }
}
