package ru.yandex.webmaster3.worker.checklist;

import java.util.Collections;
import java.util.List;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.core.zora.OfflineZoraService;
import ru.yandex.webmaster3.core.zora.ZoraConversionUtil;
import ru.yandex.webmaster3.core.zora.ZoraSourceEnum;
import ru.yandex.webmaster3.core.zora.data.fetch.ZoraFetchOpts;
import ru.yandex.webmaster3.core.zora.data.fetch.ZoraFetchResult;
import ru.yandex.webmaster3.core.zora.data.fetch.ZoraFetchStatusEnum;
import ru.yandex.webmaster3.core.zora.data.fetch.ZoraPeerVerifyMode;
import ru.yandex.webmaster3.core.zora.data.request.ZoraPDFetchRequest;
import ru.yandex.webmaster3.core.zora.data.response.ZoraRawPDFetchResponse;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.dao.HostRobotsTxtInfoYDao;
import ru.yandex.webmaster3.worker.RpsLimitedTask;
import ru.yandex.wmtools.common.sita.SitaException;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

/**
 * @author avhaliullin
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Service
public class FollowMordaRedirectsTask extends RpsLimitedTask<FollowMordaRedirectsTaskData> {
    private static final int MAX_REDIRECTS_COUNT = 7;
    private static final Duration ZORA_TIMEOUT = Duration.standardSeconds(120);
    private static final int ZORA_RETRY_ATTEMPTS = 5;
    private static final Duration ZORA_RETRY_DURATION = Duration.standardSeconds(600);

    private final HostRobotsTxtInfoYDao hostRobotsTxtInfoYDao;
    private final SiteProblemsService siteProblemsService;
    private final OfflineZoraService offlineZoraService;

    public void init() {
        setTargetRps(3.0f);
    }

    @Override
    public Result run(FollowMordaRedirectsTaskData data) throws Exception {
        WebmasterHostId hostId = data.getHostId();

        ZoraPDFetchRequest.Builder builder = ZoraPDFetchRequest.builder(IdUtils.hostIdToUrl(hostId))
                .source(ZoraSourceEnum.webmaster_robot_batch)
                .timeout(ZORA_TIMEOUT)
                .includeDocument(false)
                .includeRobots(false)
                .requestType(ZoraFetchOpts.RequestType.ONLINE)
                .sslCertPolicy(ZoraPeerVerifyMode.None)
                .redirDepth(MAX_REDIRECTS_COUNT);

        ZoraPDFetchRequest fetchRequest = builder.build();
        ZoraFetchResult fetchResult;
        try {
            fetchResult = doZoraPDFetch(hostId, fetchRequest);
        } catch (SitaException e) {
            log.error("Error response from Zora for host {}", hostId);
            throw new WebmasterException("Error response from Zora",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(),
                            "Error response from Zora"), e);
        }

        YandexHttpStatus lastLinkStatus = ZoraConversionUtil.getYandexHttpStatus(fetchResult);
        if (lastLinkStatus == YandexHttpStatus.UNKNOWN) {
            log.error("Unknown http status for host {}", hostId);
            throw new WebmasterException("Unknown http status for host",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(),
                            "Unknown http status for host"), null);
        }

        List<String> redirChain = fetchResult.getRedirChain();
        boolean hasRedirects = redirChain != null;

        ProblemSignal redirectSignal = new ProblemSignal(SiteProblemTypeEnum.MORDA_REDIRECTS, SiteProblemState.ABSENT, DateTime.now());
        ProblemSignal errorSignal;

        log.info("Host {}: http code={}, hasRedirects={}", hostId, lastLinkStatus.value(), hasRedirects);
        if (lastLinkStatus.value() >= 200 && lastLinkStatus.value() < 300 ||
                lastLinkStatus == YandexHttpStatus.HTTP_2XX) {

            log.info("Host {}: cleaning MORDA_ERROR problem", hostId);
            errorSignal = new ProblemSignal(SiteProblemTypeEnum.MORDA_ERROR, SiteProblemState.ABSENT, DateTime.now());

            if (hasRedirects) {
                WebmasterHostId lastRedirectHost = IdUtils.urlToHostId(redirChain.get(redirChain.size() - 1));
                log.info("Host {} redirects: {}, last: {}", hostId, redirChain.toArray(), lastRedirectHost);
                if (!lastRedirectHost.getPunycodeHostname().equals(hostId.getPunycodeHostname())) {
                    WebmasterHostId hostDirective = hostRobotsTxtInfoYDao.getTargetForHost(hostId);
                    log.info("Host {} redirect directive: {}", hostId, hostDirective);
                    if (hostDirective == null || !hostDirective.equals(lastRedirectHost)) {
                        log.info("Host {}: has MORDA_REDIRECT problem", hostId);
                        redirectSignal = new ProblemSignal(new SiteProblemContent.MordaRedirects(DateTime.now(), lastRedirectHost.getPunycodeHostname()), DateTime.now());
                    }
                }
            }
        } else {
            SiteProblemContent.MordaError.ExtendedStatus extendedStatus;
            boolean hasRedirectLoop = hasRedirects && hasRedirectLoop(redirChain);
            if (hasRedirectLoop) {
                log.info("Host {}: has REDIRECT_LOOP problem", hostId);
                extendedStatus = SiteProblemContent.MordaError.ExtendedStatus.REDIRECT_LOOP;
            } else {
                log.info("Host {}: has MORDA_ERROR problem", hostId);
                extendedStatus = SiteProblemContent.MordaError.ExtendedStatus.DEFAULT;
            }

            errorSignal = new ProblemSignal(
                    new SiteProblemContent.MordaError(DateTime.now(), extendedStatus, lastLinkStatus, hasRedirects), DateTime.now()
            );
        }

        siteProblemsService.updateCleanableProblem(hostId, errorSignal);
        siteProblemsService.updateCleanableProblem(hostId, redirectSignal);

        return new Result(TaskResult.SUCCESS);
    }

    private ZoraFetchResult doZoraPDFetch(WebmasterHostId hostId, ZoraPDFetchRequest req) throws SitaException, InterruptedException {
        return RetryUtils.<ZoraFetchResult, SitaException>query(RetryUtils.linearBackoff(ZORA_RETRY_ATTEMPTS, ZORA_RETRY_DURATION), () -> {
            ZoraRawPDFetchResponse rawResponse = offlineZoraService.fetchUrl(req).getResponse();
            ZoraFetchResult fetchResult = ZoraConversionUtil.verifyAndGetFirstFetchResult(rawResponse);
            if (fetchResult.getStatus() != ZoraFetchStatusEnum.FS_OK) {
                throw new SitaException("Unexpected Zora status: " + fetchResult.getStatus());
            }
            return fetchResult;
        });
    }

    private static boolean hasRedirectLoop(List<String> redirectsChain) {
        return redirectsChain.stream().anyMatch(url -> Collections.frequency(redirectsChain, url) > 1);
    }

    @Override
    public Class<FollowMordaRedirectsTaskData> getDataClass() {
        return FollowMordaRedirectsTaskData.class;
    }

}
