package ru.yandex.webmaster3.worker.checklist;

import NWebmaster.proto.urltree.Exported;
import com.google.common.collect.Range;
import org.apache.commons.io.IOUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.RobotsTxtExportInfo;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.proto.converter.RobotsTxtProtoConverter;
import ru.yandex.webmaster3.storage.util.yt.*;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * @author tsyplyaev
 */
public class RobotsTxtInfoService {
    private static final Logger log = LoggerFactory.getLogger(RobotsTxtInfoService.class);

    private YtService ytService;
    private YtPath ytRootPath;
    private String table;

    public void updateErrors(RobotsTxtStateConsumer consumer) throws InterruptedException {
        YtPath ytTablePath = YtPath.path(ytRootPath, table);

        try {
            log.info("Started importing {}", ytTablePath.getName());
            RobotsTxtInfoTableImporter importer = new RobotsTxtInfoTableImporter(ytTablePath);

            ytService.inTransaction(ytTablePath).withLock(ytTablePath, YtLockMode.SNAPSHOT).execute(cypressService -> {
                importer.importTable(cypressService, consumer);
                return true;
            });

            log.info("Finished importing {}", ytTablePath.getName());

        } catch (YtException e) {
            throw new WebmasterException("Failed to import robotx txt info table",
                    new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);

        }
    }

    public interface RobotsTxtStateConsumer {
        void accept(WebmasterHostId hostId, DateTime dateTime, RobotsTxtExportInfo info);
    }

    @Required
    public void setYtService(YtService ytService) {
        this.ytService = ytService;
    }

    @Required
    public void setYtRootPath(YtPath ytRootPath) {
        this.ytRootPath = ytRootPath;
    }

    @Required
    public void setTable(String table) {
        this.table = table;
    }

    private class RobotsTxtInfoTableImporter implements YtRowMapper<RobotsTxtInfoTableImporter.Row> {
        private static final int CHUNK_SIZE = 25_000;
        private static final int TOTAL_THREADS = 8;

        private static final String C_KEY = "key";
        private static final String C_SUBKEY = "subkey";
        private static final String C_VALUE = "value";

        private YtPath ytTablePath;
        private Row row;

        RobotsTxtInfoTableImporter(YtPath ytTablePath) {
            this.ytTablePath = ytTablePath;
            this.row = new Row();
        }

        void importTable(YtCypressService cypressService, RobotsTxtStateConsumer consumer)
                throws YtException {

            ExecutorService supplierExecutorService = Executors.newFixedThreadPool(TOTAL_THREADS);
            ExecutorService readerExecutorService = Executors.newFixedThreadPool(1);

            AsyncTableReader<Row> tableReader = new AsyncTableReader<>(cypressService, ytTablePath, Range.all(), this)
                    // читаем в случайном порядке чтобы при падени таски не страдали одни и те же пользователи в нижней части таблицы
                    .splitInRandomOrderParts(CHUNK_SIZE)
                    .inExecutor(readerExecutorService, "robots-txt-info-cacher")
                    .withRetry(7);

            try (AsyncTableReader.TableIterator<Row> it = tableReader.read()) {
                List<Callable<Void>> batch = new ArrayList<>(CHUNK_SIZE);

                while (it.hasNext()) {
                    Row row = it.next();
                    if (row.isBad()) {
                        continue;
                    }

                    final RobotsTxtExportInfo exportInfo = RobotsTxtProtoConverter.getRobotsTxtInfo(row.message);
                    final WebmasterHostId hostId = row.hostId;
                    final DateTime recordTime = row.recordTime;
                    if (exportInfo != null) {
                        //batch.add(() -> {
                            consumer.accept(hostId, recordTime, exportInfo);
                            //return null;
                        //});

                        if (batch.size() >= CHUNK_SIZE) {
                            supplierExecutorService.invokeAll(batch);
                            batch.clear();
                        }
                    }
                }

                if (!batch.isEmpty()) {
                    supplierExecutorService.invokeAll(batch);
                }
            } catch (IOException | InterruptedException e) {
                throw new YtException("Unable to import table: " + ytTablePath, e);
            } finally {
                readerExecutorService.shutdownNow();
                try {
                    readerExecutorService.awaitTermination(180, TimeUnit.SECONDS);
                } catch (InterruptedException e) {
                    // ignore, we did our best
                }

                supplierExecutorService.shutdownNow();
                try {
                    supplierExecutorService.awaitTermination(180, TimeUnit.SECONDS);
                } catch (InterruptedException e) {
                    // ignore, we did our best
                }
            }
        }

        @Override
        public void nextField(String name, InputStream data) {
            try {
                if (Objects.equals(C_KEY, name)) {
                    String urlStr = IOUtils.toString(data, StandardCharsets.UTF_8);
                    row.hostId = IdUtils.urlToHostId(urlStr);
                } else if (Objects.equals(C_SUBKEY, name)) {
                    String timestampStr = IOUtils.toString(data, StandardCharsets.UTF_8);
                    long recordTimestamp = Long.parseLong(timestampStr);
                    row.recordTime = new DateTime(recordTimestamp * 1000);
                } else if (Objects.equals(C_VALUE, name)) {
                    row.message = Exported.RobotsTxtInfo.parseFrom(data);
                }
            } catch (Exception e) {
                log.error("Unable to read host info row", e);
            }
        }

        @Override
        public Row rowEnd() {
            Row r = row;
            row = new Row();

            return r;
        }

        @Override
        public List<String> getColumns() {
            return Arrays.asList(C_KEY, C_SUBKEY, C_VALUE);
        }

        class Row {
            WebmasterHostId hostId;
            Exported.RobotsTxtInfo message;
            DateTime recordTime;

            boolean isBad() {
                return hostId == null || message == null;
            }
        }
    }
}

