package ru.yandex.webmaster3.worker.checklist;

import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.dao.HostRobotsTxtInfoYDao;
import ru.yandex.webmaster3.storage.host.data.HostRobotsTxtInfo;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.*;

/**
 * @author tsyplyaev
 */

@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component
public class RobotsTxtInfoUpdateTask extends PeriodicTask<PeriodicTaskState> {
    private static final Logger log = LoggerFactory.getLogger(RobotsTxtInfoUpdateTask.class);
    private static final int BATCH_SIZE = 2048;
    private final RobotsTxtInfoService robotsTxtInfoService;
    private final SiteProblemsService siteProblemsService;
    private final HostRobotsTxtInfoYDao hostRobotsTxtInfoYDao;

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.ROBOTS_TXT_INFO_UPDATE;
    }

    @Override
    public Result run(UUID runId) throws Exception {
        Map<WebmasterHostId, List<ProblemSignal>> mapNoRobots = new HashMap<>(BATCH_SIZE);
        Map<WebmasterHostId, List<ProblemSignal>> mapErrorsInRobots = new HashMap<>(BATCH_SIZE);

        List<HostRobotsTxtInfo> batch = new ArrayList<>();
        robotsTxtInfoService.updateErrors((hostId, dateTime, info) -> {
            try {
                ProblemSignal noRobots;
                ProblemSignal errorsInRobots;
                try {
                    WebmasterHostId targetHost = info.getHostDirective() == null ? null : IdUtils.urlToHostId(info.getHostDirective());
                    HostRobotsTxtInfo robotsTxtInfo = new HostRobotsTxtInfo(hostId, info.isRobotsTxtLoaded(), targetHost);
                    batch.add(robotsTxtInfo);
                    if (batch.size() >= BATCH_SIZE) {
                        hostRobotsTxtInfoYDao.updateInfos(batch);
                        batch.clear();
                    }
                } catch (Exception e) {
                    log.info("Bad Host directive '{}', '{}' for host {}", info.getHostDirective(), info.getHostDirective2NdLevel(), hostId);
                }

                if (!info.isRobotsTxtLoaded()) {
                    noRobots = new ProblemSignal(new SiteProblemContent.NoRobotsTxt(), dateTime);
                    errorsInRobots = new ProblemSignal(SiteProblemTypeEnum.ROBOTS_TXT_ERROR, SiteProblemState.UNDEFINED, dateTime);
                } else {
                    noRobots = new ProblemSignal(SiteProblemTypeEnum.NO_ROBOTS_TXT, SiteProblemState.ABSENT, dateTime);
                    if (info.isHasErrors() || info.isHasWarnings()) {
                        errorsInRobots = new ProblemSignal(new SiteProblemContent.RobotsTxtError(), dateTime);
                    } else {
                        errorsInRobots = new ProblemSignal(SiteProblemTypeEnum.ROBOTS_TXT_ERROR, SiteProblemState.ABSENT, dateTime);
                    }
                }

                mapNoRobots.put(hostId, List.of(noRobots));
                mapErrorsInRobots.put(hostId, List.of(errorsInRobots));
                if (mapNoRobots.size() > BATCH_SIZE) {
                    siteProblemsService.updateRealTimeSiteProblems(mapNoRobots, SiteProblemTypeEnum.NO_ROBOTS_TXT.getStorageType());
                    siteProblemsService.updateRealTimeSiteProblems(mapErrorsInRobots, SiteProblemTypeEnum.ROBOTS_TXT_ERROR.getStorageType());
                    mapNoRobots.clear();
                    mapErrorsInRobots.clear();
                }
            } catch (WebmasterException e) {
                log.error("Failed to update robots txt problems for host {}", hostId, e);
            } catch (WebmasterYdbException e) {
                log.error("Failed to update robots txt info for host {}", hostId, e);
            }
        });

        if (!batch.isEmpty()) {
            hostRobotsTxtInfoYDao.updateInfos(batch);
        }

        siteProblemsService.updateRealTimeSiteProblems(mapNoRobots, SiteProblemTypeEnum.NO_ROBOTS_TXT.getStorageType());
        siteProblemsService.updateRealTimeSiteProblems(mapErrorsInRobots, SiteProblemTypeEnum.ROBOTS_TXT_ERROR.getStorageType());
        mapNoRobots.clear();
        mapErrorsInRobots.clear();
        return new Result(TaskResult.SUCCESS);
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 9 * * *");
    }
}
