package ru.yandex.webmaster3.worker.checklist;

import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.node.BooleanNode;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Range;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserDomainMessageEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.storage.yql.YqlService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by Oleg Bazdyrev on 28/06/2021.
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class SendRepeatedChecklistNotificationsTask extends PeriodicTask<PeriodicTaskState> {

    private static final String ATTR_TABLE_PROCESSED = "processed";
    private static final String PREPARE_REPEATED_NOTIFICATIONS_QUERY = """
            use arnold;
            pragma yt.DefaultMaxJobFails = '1';
            pragma yson.DisableStrict;

            $fixScheme = Re2::Replace("://");
            $getScheme = ($url) -> {
            return $fixScheme(NVL(Url::GetScheme($url), "http"), "");
            };
            $getPort = ($url) -> {
            return NVL(Url::GetPort($url), IF($getScheme($url) == "http", 80, 443));
            };
            $url2HostId = ($url) -> {
            return $getScheme($url) || ":" || Url::GetHost($url) || ":" || YQL::ToString($getPort($url));
            };

            $currentDate = CurrentUtcDate();
            $halfYearBack = CurrentUtcDate() - Interval('P180D');
            $printDate = DateTime::Format('%Y-%m-%d');

            $notifications = (
                SELECT DateTime::MakeDate(DateTime::ParseIso8601(iso_eventtime)) as EventDate,
                    Yson::ConvertToString(content.hostId) as HostId,
                    Yson::ConvertToString(content.messageContent.problemType) as ProblemType,
                    Yson::ConvertToString(content.notificationType) as NotificationType,
                    content.messageContent as MessageContent,
                    nvl(Yson::ConvertToInt64(content.messageContent.attempt), 0) as Attempt,
                FROM range(`${DAILY_EVENTS_LOG_PATH}`, $printDate($halfYearBack))
                WHERE type == 'USER_HOST_MESSAGE' and Yson::ConvertToString(content.messageContent.type) == 'CHECKLIST_CHANGES'
            );

            $alerts = (
                SELECT HostId, ProblemType, LastUpdate, ActualSince
                FROM `${SITE_PROBLEMS_TABLE}`
                GROUP BY $url2HostId(Host) as HostId, ProblemType, LastUpdate, ActualSince
            );

            $repeated_notifications = (
                SELECT HostId, ProblemType, NotificationType, NewEventDate as EventDate, MessageContent, Attempt
                FROM range(`${REPEATED_NOTIFICATIONS_PATH}`, $printDate($halfYearBack))
            );

            $all_notifications = (
                SELECT HostId,
                    ProblemType,
                    NotificationType,
                    max(EventDate) as EventDate,
                    max_by(MessageContent, EventDate) as MessageContent,
                    max_by(Attempt, EventDate) as Attempt
                FROM (
                    SELECT * FROM $notifications
                    UNION ALL
                    SELECT * FROM $repeated_notifications
                )
                GROUP BY HostId, ProblemType, NotificationType
            );

            -- candidates for repeated notification
            $currentNotificationsTable = '${REPEATED_NOTIFICATIONS_PATH}' || '/' || $printDate($currentDate);
            INSERT INTO $currentNotificationsTable WITH TRUNCATE\s
            SELECT
                a.HostId as HostId,
                LastUpdate,
                ActualSince,
                a.ProblemType as ProblemType,
                NotificationType,
                MessageContent,
                Attempt + 1 as Attempt,
                EventDate as PrevEventDate,
                $currentDate as NewEventDate,
                DateTime::ToDays($currentDate - EventDate) as Age
            FROM $all_notifications as n
            INNER JOIN $alerts as a USING (HostId, ProblemType)
            WHERE
                n.Attempt < 3 AND
                (n.NotificationType == 'SITE_PROBLEM_FATAL' and DateTime::ToDays($currentDate - EventDate) >= 7) OR
                (n.NotificationType == 'SITE_PROBLEM_CRITICAL' and DateTime::ToDays($currentDate - EventDate) >= 14) OR
                (n.NotificationType == 'SITE_PROBLEM_POSSIBLE' and DateTime::ToDays($currentDate - EventDate) >= 30) OR
                (n.NotificationType == 'SITE_PROBLEM_RECOMMENDATION' and DateTime::ToDays($currentDate - EventDate) >= 90);
            """;

    private final AbtService abtService;
    private final YqlService yqlService;
    private final YtService ytService;
    private final WMCEventsService wmcEventsService;
    @Value("${external.yt.service.arnold.root.default}/export/checklist/site-problems")
    private YtPath siteProblemsTable;
    @Value("${external.yt.service.arnold}://home/logfeller/logs/webmaster-prod-processed-events-log/1d")
    private YtPath dailyEventsLogPath;
    @Value("${external.yt.service.arnold.root.default}/export/checklist/repeated-notifications")
    private YtPath repeatedNotificationsPath;

    @Override
    public Result run(UUID runId) throws Exception {
        // проверим наличие вчерашних логов
        ytService.withoutTransaction(cypressService -> {
            YtPath yesterdayLogTable = YtPath.path(dailyEventsLogPath, LocalDate.now().minusDays(1).toString());
            if (!cypressService.exists(yesterdayLogTable)) {
                throw new IllegalStateException("Yesterday events log table is missing, aborting");
            }
            // проверим, нет ли уже сегодняшней таблицы
            YtPath currentDayNotificationsTable = YtPath.path(repeatedNotificationsPath, LocalDate.now().toString());
            if (!cypressService.exists(currentDayNotificationsTable)) {
                prepareRepeatedNotifications();
            } else {
                YtNode currentTableNode = cypressService.getNode(currentDayNotificationsTable);
                if (currentTableNode.getNodeMeta().has(ATTR_TABLE_PROCESSED)) {
                    log.info("Table already processed and notifications was sent. Exiting");
                    return true;
                }
            }
            sentRepeatedNotifications(cypressService, currentDayNotificationsTable);
            cypressService.set(YtPath.attribute(currentDayNotificationsTable, ATTR_TABLE_PROCESSED), BooleanNode.TRUE);
            return true;
        });

        return new Result(TaskResult.SUCCESS);
    }

    private void prepareRepeatedNotifications() {
        log.info("Preparing repeated notifications");
        // генерируем свежую пачку данных для отправки
        StrSubstitutor substitutor = new StrSubstitutor(ImmutableMap.of(
                "DAILY_EVENTS_LOG_PATH", dailyEventsLogPath.toYtPath(),
                "SITE_PROBLEMS_TABLE", siteProblemsTable.toYtPath(),
                "REPEATED_NOTIFICATIONS_PATH", repeatedNotificationsPath.toYtPath()
        ));
        String query = substitutor.replace(PREPARE_REPEATED_NOTIFICATIONS_QUERY);
        yqlService.execute(query);
    }

    private void sentRepeatedNotifications(YtCypressService cypressService, YtPath table) {
        log.info("Sending repeated notifications from {}", table);
        AsyncTableReader<RepeatedNotificationRow> tableReader = new AsyncTableReader<>(
                cypressService, table, Range.all(), YtTableReadDriver.createYSONDriver(RepeatedNotificationRow.class))
                .withThreadName("repeated-notifications-reader");
        try (var iterator = tableReader.read()) {
            while (iterator.hasNext()) {
                RepeatedNotificationRow row = iterator.next();
                // only processing CRITICAL and FATAL
                if (row.getNotificationType() != NotificationType.SITE_PROBLEM_CRITICAL &&
                        row.getNotificationType() != NotificationType.SITE_PROBLEM_POSSIBLE) {
                    continue;
                }
                // only processing host with exp
                if (abtService.isInExperiment(row.getHostId(), Experiment.REPEATED_NOTIFICATIONS)) {
                    // refresh dates and attempt on checlist
                    DateTime lastUpdate = row.getLastUpdate() == null ? null : new DateTime(row.getLastUpdate());
                    DateTime actualSince = row.getActualSince() == null ? null : new DateTime(row.getActualSince());
                    MessageContent.ChecklistChanges changes = new MessageContent.ChecklistChanges(row.getHostId(), row.getProblemType(), lastUpdate,
                            actualSince, row.getMessageContent().getProblemContent(), row.getAttempt());
                    if (row.getProblemType().isTurboProblem()) {
                        wmcEventsService.addEvent(new RetranslateToUsersEvent<>(UserDomainMessageEvent.create(
                                row.getHostId(), changes, row.getNotificationType(), row.getProblemType().isCriticalSeverity())));
                    } else {
                        wmcEventsService.addEvent(new RetranslateToUsersEvent<>(UserHostMessageEvent.create(
                                row.getHostId(), changes, row.getNotificationType(), row.getProblemType().isCriticalSeverity())));
                    }
                }
            }
        } catch (Exception e) {
            throw new WebmasterException("Error reading notifications table", new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
        }

    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.SEND_REPEATED_CHECKLIST_NOTIFICATIONS;
    }

    @Override
    public TaskSchedule getSchedule() {
        // ежедневно по утрам
        return TaskSchedule.startByCron("0 5 8 * * *");
    }

    @lombok.Value
    public static class RepeatedNotificationRow {
        @JsonProperty("HostId")
        WebmasterHostId hostId;
        @JsonProperty("ProblemType")
        SiteProblemTypeEnum problemType;
        @JsonProperty("NotificationType")
        NotificationType notificationType;
        @JsonProperty("ActualSince")
        Long actualSince;
        @JsonProperty("LastUpdate")
        Long lastUpdate;
        @JsonProperty("MessageContent")
        MessageContent.ChecklistChanges messageContent;
        @JsonProperty("PrevEventDate")
        LocalDate prevEventDate;
        @JsonProperty("NewEventDate")
        LocalDate newEventDate;
        @JsonProperty("Attempt")
        int attempt;
    }
}
